\ignore{
\begin{code}
{-# LANGUAGE EmptyDataDecls,
    MultiParamTypeClasses, FunctionalDependencies, UndecidableInstances,
    FlexibleInstances, RankNTypes, FlexibleContexts,
    TypeFamilies, GADTs, ScopedTypeVariables, Rank2Types,  ConstraintKinds,
    TemplateHaskell, TypeOperators, QuasiQuotes
  #-}

module Examples where

import Prelude hiding ((*), (/), (+), negate)
import qualified Prelude as P ((*), (/), (+), negate)

import NumericTypes
import FloatingPoint
import TypeLevelList
import SIUnits 
import qualified SIUnits as U
import UnitDefinitions
import TypeGen

import Data.List (transpose)
import Data.Complex hiding ((:+))
import qualified Numeric.FFT as FFT

\end{code}
}


\section{Examples}\label{sec:Examples}

A collection of examples, starting from simple and ending in complex. 


\subsection{Units}\label{subsec:Units}

Examples of the use of code in \hask{SIUnits}.

\begin{code}

eg1 = SIUnitVal 7 :: SIUnitVal M1 Kg0 S1 A1 Mol1 K0 Double
eg2 = SIUnitVal 8 :: SIUnitVal M1 Kg1 S1 A1 Mol1 K0 Double
eg3 = SIUnitVal 8 :: SIUnitVal Kg2 M1 S1 A1 Mol1 K0 Double

distance = SIUnitVal 2 :: Meter Double
time =     SIUnitVal 10 :: Second Double
velocity = SIUnitVal 0.2 :: MPerS Double 
testval =  SIUnitVal 4 :: Unitless Double


\end{code}

A simple function for calculation potential energy in the Earth's gravity given change in height and mass.
\begin{code}

potentialEnergy ::          Kilogram Double -> 
                            Meter Double -> 
                            Joule Double
potentialEnergy mass height = mass * g * height
    where g = SIUnitVal 9.81 :: SIUnitVal M1 Kg0 S_2 A0 Mol0 K0 Double
    
\end{code}
The following works correctly: 
\begin{fakecode}
potentialEnergy (SIUnitVal 3.0 :: Kilogram Double) (SIUnitVal 25.0 :: Meter Double)

735.75 m^{2} kg s^{-2} :: Joule Double
\end{fakecode}
While an incorrectly typed arguement will yield: 
\begin{fakecode}
potentialEnergy (SIUnitVal 1.0 :: Second Double) (SIUnitVal 1.0 :: Unitless Double)

    Couldn't match expected type `Kg1' with actual type `Kg0'
    Expected type: Kilogram Double
      Actual type: Second Double
    In the first argument of `potentialEnergy', namely
      `(SIUnitVal 1.0 :: Second Double)'
    In the expression:
      potentialEnergy
        (SIUnitVal 1.0 :: Second Double) (SIUnitVal 1.0 :: Unitless Double)
\end{fakecode}


\subsection{Numeric functions}\label{subsec:NumericFunctions}

A trivial function that demonstrates the functionality of the \hask{NonZero} class.
\begin{code}
toIntNonZero :: forall a . (NonZero a, Nat a) => a -> Integer
toIntNonZero _ = integerValue (undefined :: a) 
\end{code}

\begin{fakecode}
toIntNonZero (undefined :: SIZE3 D0 D4 D2)
42

toIntNonZero (undefined :: SIZE3 D0 D0 D0)
    No instance for (Fail TypeLevelNumIsZero)
      arising from a use of `toIntNonZero'
    Possible fix: add an instance declaration for (Fail ())

\end{fakecode}

We define a function \hask{mult10} which demonstrates multiplication of type level numbers.
Some sample type level numbers are also defined. 
\begin{code}
mult :: MultD a b c => a -> b -> c
mult _ _ = undefined

tl_16  = [num| 16. |]
tl_3   = [num| 3.  |]
tl_5e5 = [num| 5e5 |]
tl_3e8 = [num| 3e8 |]
\end{code}

GHCI gives the following results, as expected.
\begin{fakecode}
:type mult10 tl_16 tl_3
mult10 tl_16 tl_3 :: SIZE D0 D0 D0 D0 D0 D0 D0 D0 D4 D8

:type mult10 tl_5e5 tl_3
mult10 tl_5e5 tl_3 :: SIZE D0 D0 D0 D0 D1 D5 D0 D0 D0 D0
\end{fakecode}


\subsection{Frames}\label{subsec:Frames}

The examples below demonstrate an application of type level numbers and units to the Fast Fourier Transform.\\ 
The basic class defintion for a frame of reference, containing a function to show the name of a frame,
and a type alias for the base unit of the frame. 
The type of \hask{BaseUnit} should be an instance of \hask{SIUnitVal} with an empty tuple as a value.
\begin{code}

class Frame frame where
  type BaseUnit frame
  name :: frame -> String

\end{code}


One application of frames of reference is to the Fast Fourier Transform, which has an inverse operation. 
The implementation of FFT is provided by the package pure-fft. The package contains two functions useful to us; 
hask{fft}, which is the FFT and \hask{ifft}, which is the inverse operation. 
We make the assertions that inverse FFT applied to the result of FFT has the same type as the original arguement 
(equivalently, \hask{ ( (ifft . fft) (a :: t) ) :: t }). 
We let the user assert that two frames are dual to each other, and check that the units match.
The functional depency is asserting that the dual of the dual frame is the original frame.
This has to be argued outside the type system, with a definition of what the dual frame is.
Maybe we need to argue it in the context of discretizations, 
if there is no notion of dual coordinate frames, but there is a notion of dual vector spaces.

Before dual frames can be formalized, we first formalize dual units; these are units whose dimensions are reciprocals, or whose product is unitless.
Instances of the class which defines dual frames of reference must contain two frames whose units are dual.
\begin{code}

class DualUnits a b | a -> b, b -> a where
instance (Frame a, Frame b, U.Mult (BaseUnit a) (BaseUnit b) (U.Unitless ())) => DualUnits a b where


class (DualUnits a b) => AssertDualFrames a b | a -> b, b -> a where

\end{code}

The user must then define instances of \hask{Frame} for each frame of reference. 
They are responsible for defining \hask{BaseUnit}s which are consistent; 
they then must create an instance of \hask{AssertDualFrames} to guarantee that the frames are in fact dual to each other.
\begin{code}
data LabFrame
instance Frame LabFrame where
  type BaseUnit LabFrame = (U.Meter ())
  name _ = "LabFrame"
  
data LabFrameT
instance Frame LabFrameT where
  type BaseUnit LabFrameT = (U.PerM ())
  name _ = "LabFrameDual"

instance AssertDualFrames LabFrame LabFrameT where

\end{code}

If the user defines a frame improperly and attempts to assert the duality of the improperly defined frames
\begin{code}
data BadFrame
instance Frame BadFrame where
  type BaseUnit BadFrame = (U.Unitless ())
  name _ = "LabFrameDual"
\end{code}
\begin{fakecode}
instance AssertDualFrames LabFrame BadFrame where
\end{fakecode}
they will meet this compile time error
\begin{fakecode}
    Couldn't match type `UnitDefinitions.M0' with `UnitDefinitions.M1'
    When using functional dependencies to combine
      UnitDefinitions.AddUnit
        UnitDefinitions.M0 UnitDefinitions.M0 UnitDefinitions.M0,
        arising from the dependency `b c -> a'
\end{fakecode}
The error is clear that there doesn't exist an instance for adding meters to meters to get meters. 



We defining phantom type wrapper for the \hask{Size} class just for better representation. We also define helper functions for each phantom type which will take a phantom type and give the underlying \hask{Size} instance. These wrappers will be used to define more readable datatypes for demonstration purposes below. The first will be used to represent the number of samples in a discretization; the second and third will represent the numerator and denominator, respectively, of the step size.
\begin{code}

class Wrapper w where 
    unwrap :: w s -> s
    unwrap = undefined

class NumSamplesC n where 
data NumSamples size where
    NumSamples :: (Nat size, NonZero size) => size -> NumSamples size 
instance (Nat size, NonZero size) => NumSamplesC (NumSamples size) where
instance Wrapper NumSamples where 

class StepSizeC s where 
data StepSize size where
    StepSize :: (FloatT size, NonZero size) => size -> StepSize size 
instance (FloatT size, NonZero size) => StepSizeC (StepSize size) where
instance Wrapper StepSize where 





\end{code}

We define another sample frame.
\begin{code}

data CanalFrame
instance Frame CanalFrame where
  type BaseUnit CanalFrame = (U.Meter ())
  name _ = "CanalFrame"
\end{code}



We are now ready to define a datatype to represent a series of values belonging to one dimensional discretization. The discretization has several physical properties which are all present in the constraints of the datatype.
\begin{code}
data Discretization1D a b c d e where
    Discretization1D :: 
        (Frame frame, U.SIUnit (BaseUnit frame), U.SIUnit rangeU , 
         NumSamplesC (NumSamples numSamples), 
         StepSizeC (StepSize stepSize)
        ) => val -> Discretization1D frame
                                    (NumSamples numSamples)
                                    (StepSize stepSize)
                                    rangeU 
                                    val


\end{code}


An instance for addition and subtraction of discretization is implemented.
\begin{code}
instance (U.Add val) => U.Add (Discretization1D frame numSamples stepSize rangeU val) where
  (Discretization1D a) + (Discretization1D b) = Discretization1D (a U.+ b)
  (Discretization1D a) - (Discretization1D b) = Discretization1D (a U.- b)
  negate (Discretization1D a) = Discretization1D (U.negate a)
\end{code}


Using examples below, we can see that addition between discretizations which do not share all of the same properties produces errors.
\begin{code}
meas1 :: Discretization1D  LabFrame 
                           (NumSamples [nat| 4 |] ) 
                           (StepSize [num| 1/500 |]) 
                           (U.Meter ()) 
                           [Complex Double]
meas1 = Discretization1D [0,1,2,3]

meas2 :: Discretization1D  LabFrame 
                           (NumSamples [nat| 5 |])
                           (StepSize [num| 1/500 |] )
                           (U.Second ()) 
                           [Complex Double]
meas2 = Discretization1D [0.4,0.8,1.2,1.6]

canalSample1 :: Discretization1D CanalFrame 
                                 (NumSamples [nat| 12 |])
                                 (StepSize [num| 1/100 |])
                                 (U.Meter ()) 
                                 [Double]
canalSample1 = Discretization1D [1, 1.2, 1.3, 1.12, 1.23, 1.12, 1.15, 1.25, 1.18, 1.20, 1.24, 1.28]

canalSample2 :: Discretization1D CanalFrame 
                                 (NumSamples [nat| 12 |])
                                 (StepSize [num| 1/50 |])
                                 (U.Meter ()) 
                                 [Double]
canalSample2 = Discretization1D [1, 1.21, 1.2, 1.42, 1.3, 1.32, 1.12, 1.25, 1.23, 1.20, 1.12, 1.28]
\end{code}

\begin{fakecode}
:t meas1 U.+ canalSample1
    Couldn't match expected type `LabFrame'
                with actual type `CanalFrame'
                
:t canalSample2 U.+ canalSample1
    Couldn't match expected type `D2' with actual type `D1'
    Expected type: Discretization1D
    
:t meas1 U.+ meas2
    Couldn't match expected type `UnitDefinitions.M1'
                with actual type `UnitDefinitions.M0'
    
\end{fakecode}

We can now use the above defintions to define a class \hask{FT} which will be used to describe the relationship between a set of discrete data and the Fourier Transform of that data. The functional dependencies indicate that the intial data uniquely determines the Fourier transform, and the result of the Fourier transform uniquely determines the initial data.
\begin{code}
class FT a b | a -> b, b -> a where
  ft :: a -> b 
  invFt :: b -> a
\end{code}

The instance of \hask{FT} for one dimensional discretizations constrains the frames of reference of the original data and the result as being dual frames. 
One requirement of Fourier transform is the product of the step sizes of input and output data and the number of samples must be one. The \hask{MultDNonZero} class is used to apply this constraint. 
Note that in order to more easily generalize \hask{FT} to instances of \hask{Size} which have a width other than three, it would be helpful to define a special type \hask{One}. 
\begin{code}
instance (
    AssertDualFrames frame1 frame2, DualUnits frame1 frame2,
    StepSizeC (StepSize stepSize1), 
    StepSizeC (StepSize stepSize2), 
    NumSamplesC (NumSamples numSamp),
    Frame frame1, U.SIUnit (BaseUnit frame1),
    Frame frame2, U.SIUnit (BaseUnit frame2),
--    ToFloat numSamp numSampF,
--    MultD stepSize1 numSampF t0 , MultD t0 stepSize2 z,  IsOne z TTrue -- this doesnt seem to work
    ) => 
    FT (Discretization1D frame1 (NumSamples numSamp) (StepSize stepSize1) rangeU [Complex Double]) 
       (Discretization1D frame2 (NumSamples numSamp) (StepSize stepSize2) rangeU [Complex Double]) 
        where
            ft (Discretization1D x) = (Discretization1D $ FFT.fft x)
            invFt (Discretization1D x) = (Discretization1D $ FFT.ifft x)
\end{code}

We can now apply Fourier transform to a data set with type safety. The type system infers the type of the result; the result itself is not of particular interest, however.
\begin{fakecode}
:t meas1
meas1 :: Discretization1D  LabFrame 
                           (NumSamples (SIZE3 D0 D0 D4))
                           (StepSizeNum(SIZE3 D0 D0 D1))
                           (StepSizeDenom(SIZE3 D5 D0 D0)) 
                           (U.Meter ()) 
                           [Complex Double] 
:t ft meas1
ft meas1
  :: Discretization1D
       LabFrameT
       (NumSamples (SIZE3 D0 D0 D4))
       (StepSizeNum (SIZE3 D5 D0 D0))
       (StepSizeDenom (SIZE3 D0 D0 D4))
       (U.Meter ())
       [Complex Double]
\end{fakecode}
The step sizes of \hask{meas1} and \hask{ft meas1} are $(1/500) = 0.002$ and $(500/4) = 125$, respectively. We indeed have that $0.002 \times 125 \times 4 = 1$, as the constraint on the Fourier transform requires.


We can define a similair rectangular discretization in two dimensions. In this case, we need to indicate the number of samples in both perpendicular basis directions, as well as the step size. 
\begin{code}

data LabFrame2D
data LabFrame2DT

instance Frame LabFrame2D where
  type BaseUnit LabFrame2D = U.Meter ()
  name _ = "lab2"
  
instance Frame LabFrame2DT where
  type BaseUnit LabFrame2DT = U.PerM ()
  name _ = "lab2t"
  
instance AssertDualFrames LabFrame2D LabFrame2DT

data Discretization2D a b c d e f g where
        Discretization2D :: (Frame frame , U.SIUnit rangeU, U.SIUnit (BaseUnit frame), 
                             NumSamplesC (NumSamples numSampX), StepSizeC (StepSize stepSizeX),
                             NumSamplesC (NumSamples numSampY), StepSizeC (StepSize stepSizeY)
                            ) => val -> 
            Discretization2D frame
                             (NumSamples numSampX)
                             (StepSize stepSizeX)
                             (NumSamples numSampY)
                             (StepSize stepSizeY)
                             rangeU 
                             val

labSample2D :: Discretization2D LabFrame2D                   
                                (NumSamples [num| 8    |])   
                                (StepSize   [num| 5e-2 |])        
                                (NumSamples [num| 4    |])           
                                (StepSize   [num| 1e-2 |])            
                                (U.Meter ())                      
                                [[Complex Double]] 
labSample2D = Discretization2D
                    [[-0.2 ,-0.26 ,-0.32 ,-0.38 ,-0.44 ,-0.5 ,-0.56 ,-0.63]
                    ,[0.2  ,0.26  ,0.32  ,0.38  ,0.44  ,0.5  ,0.56  ,0.76 ]
                    ,[0.6  ,0.78  ,0.96  ,1.14  ,1.32  ,1.5  ,1.68  ,1.98 ]
                    ,[1.0  ,1.3   ,1.6   ,1.90  ,2.2   ,2.5  ,2.8   ,2.94 ]]

instance ( U.SIUnit rangeU, 
          Frame frame1 , U.SIUnit (BaseUnit frame1), 
          Frame frame2 , U.SIUnit (BaseUnit frame2), 
          NumSamplesC (NumSamples numSampX), NumSamplesC (NumSamples numSampY),
          StepSizeC (StepSize stepSizeX1), StepSizeC (StepSize stepSizeY1),
          StepSizeC (StepSize stepSizeX2), StepSizeC (StepSize stepSizeY2),
          MultD stepSizeX1 numSampX t0, MultD t0 stepSizeX2 z, IsOne z TTrue,
          MultD stepSizeY1 numSampY q0, MultD q0 stepSizeY2 w, IsOne w TTrue
        )
   => FT (Discretization2D  frame1
                            (NumSamples numSampX)
                            (StepSize stepSizeX1)
                            (NumSamples numSampY)
                            (StepSize stepSizeY1)
                            rangeU
                            [[Complex Double]])
         (Discretization2D  frame2
                            (NumSamples numSampX)
                            (StepSize stepSizeX2)
                            (NumSamples numSampY)
                            (StepSize stepSizeY2)
                            rangeU 
                            [[Complex Double]])
   where
 ft (Discretization2D x) = 
     (Discretization2D . transpose . (map FFT.fft ) . transpose . (map FFT.fft )) x
 invFt (Discretization2D x) = 
     (Discretization2D . transpose . (map FFT.ifft) . transpose . (map FFT.ifft)) x

\end{code}

