\ignore{
\begin{code}
{-# LANGUAGE 
    RankNTypes, 
    ScopedTypeVariables, 
    EmptyDataDecls, 
    MultiParamTypeClasses,
    FunctionalDependencies, 
    FlexibleInstances, 
    UndecidableInstances, 
    OverlappingInstances, 
    TypeFamilies , 
    ImpredicativeTypes, 
    FlexibleContexts, 
    DeriveDataTypeable,
    IncoherentInstances,
    ConstraintKinds,
    TypeOperators,
    GADTs
    #-}
module FloatingPoint where

import NumericTypes
import TypeLevelList
import Data.Ratio ((%))
import GHC.Real (Ratio(..))
import System.IO.Unsafe
import Unsafe.Coerce

\end{code}
}

\section{Floating point numbers}\label{sec:Floating}
The implementation of floating point numbers and basic functions on them. 


The integer datatype. \hask{INT} takes a sign and a natural number. 
\begin{code}
data INT s n where
    INT :: (Sign s, Nat n) => s -> n -> INT s n

\end{code}
The mantissa is the fractional digits after a leading '0.'
So \hask{((INT (:+) (D3 :$ D1 :$ D4 :$ T)) `EXP` (INT (:+) (D1 :$ T))) = (+ 0.314) * 10 ^ (+ 1) = 3.14}
This interpretation makes comparison and addition much simpler.
Two constructors exist; they do the exact same thing.
\begin{code}
data FL base exp where
    FL :: (INT s0 n0) -> (INT s1 n1) -> FL (INT s0 n0) (INT s1 n1)
    EXP   :: (INT s0 n0) -> (INT s1 n1) -> FL (INT s0 n0) (INT s1 n1)

\end{code}
A generic number instance, and a show instance, for integers.
\begin{code}
instance (Sign s, Nat n) => GenericNumber (INT s n) where 
    type One  (INT s n) = (INT (:+) (D1 :$ T))
    type Zero (INT s n) = (INT (:+) (D0 :$ T))
    numToRational _ = ((sgnVal (undefined :: s)) * (integerValue (undefined :: n))) % 1
    numToIntegral _ = fromIntegral $ (sgnVal (undefined :: s)) * (integerValue (undefined :: n))
instance (GenericNumber (INT s n)) => Show (INT s n) where show = show . ( numToIntegral :: (INT s n) -> Integer )


\end{code}
A generic number instance, and a show instance, for floats.
\begin{code}
instance (Sign s0, Sign s1, Nat n0, Nat n1, EqualizeLists D0 (n0, T) (n0n, d), Nat d, Nat d2, d2 ~ (D1 :$ d)) => GenericNumber (FL (INT s0 n0) (INT s1 n1)) where 
    type One  (FL (INT s0 n0) (INT s1 n1)) = FL (INT (:+) (D1 :$ T)) (INT (:+) (D1 :$ T))
    type Zero (FL (INT s0 n0) (INT s1 n1)) = FL (INT (:+) (D0 :$ T)) (INT (:+) (D1 :$ T))
    
\end{code}
Since \hask{fromIntegral} is not aware of the implicit decimal place in the float,
the data level number must be divided by the correct amount.
\begin{code}
    numToFloating _ = m  * (10 ** (fromIntegral ((numToIntegral (undefined :: (INT s1 n1)))))) / d
        where m = (fromIntegral (numToIntegral (undefined :: (INT s0 n0))))
              d = (fromIntegral (integerValue (undefined :: d2)))    
    numToRational _ = (m % d) * ce 
        where m = numToIntegral (undefined :: (INT s0 n0))
              d = integerValue (undefined :: d2)
              e = numToIntegral (undefined :: (INT s1 n1)) 
              ce = if e < 0 then 1 % (10 ^ (e * (-1))) else (10 ^ e) % 1

\end{code}
Because of the representation, leading zeroes in front of the mantissa have significance,
while the integral representation throws them out.
So we have to cheat and prepend a one, which preserves the zeros, and later remove it.
In order to show the number in scientific notation, we need to place the decimal after the first non-zero digit. 
The number will not be in this form, so we must change the exponent by the appropriate amount.
\begin{code}
instance (Sign s0, Sign s1, Nat n0, Nat n1, GenericNumber (FL (INT s0 n0) (INT s1 n1)), np ~ (D1 :$ n0), Nat np
    ) => Show (FL (INT s0 n0) (INT s1 n1)) where 
        show _ 
            | (length dropZ) == 0 = sgn ++ "0"
            | (length dropZ) < 2 = sgn ++ dropZ ++ "e" ++ (show exp)
            | otherwise = sgn ++ (insertAt "." 1 dropZ) ++ "e" ++ (show exp)
                where insertAt c n xs = (take n xs) ++ c ++ (drop n xs)
                      man = integerValue (undefined :: np)
                      exp' = numToIntegral (undefined :: (INT s1 n1))
                      exp = exp' - (fromIntegral numZ)
                      sgn = if (sgnVal (undefined :: s0)) < 0 then "-" else ""
                      numZ = 1 + (length $ takeWhile (=='0') (tail (show man)))
                      dropZ = dropWhile (=='0') (tail (show man))

\end{code}
User readable numbers.
\begin{code}
data TimesTenToPow s n where
    TimesTenToPow :: (Sign s, Nat n) => s -> n -> TimesTenToPow s n

data ZeroPoint base exp where
    ZeroPoint :: Nat base => base -> TimesTenToPow sign exp -> ZeroPoint base (TimesTenToPow sign exp)

data FLOAT s n where
    FLOAT :: Sign s => s -> ZeroPoint base exp -> FLOAT s (ZeroPoint base exp)

instance (FloatMarshall q (FLOAT a b), Show q
    ) => (Show (FLOAT a b)) where
      show _ = show (undefined :: q)

instance (FloatMarshall q (FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))), GenericNumber q) => GenericNumber (FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))) where
    type One  (FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))) = (FLOAT Pos (ZeroPoint (D1 :$ T) (TimesTenToPow Pos (D1 :$ T))))
    type Zero (FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))) = (FLOAT Pos (ZeroPoint (D0 :$ T) (TimesTenToPow Pos (D1 :$ T))))
    numToFloating _ = numToFloating (undefined :: q)
    numToRational _ = numToRational (undefined :: q)
    numToIntegral _ = undefined

\end{code}
Marshalling between internal and external representations of floats.
\begin{code}
class FloatMarshall a b | a -> b, b -> a where
    f1Tof2 :: a -> b
    f2Tof1 :: b -> a
instance FloatMarshall (FL (INT s0 n0) (INT s1 n1)) (FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))) where
    f1Tof2 (FL (INT s0 n0) (INT s1 n1)) =   FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))
    f1Tof2 (EXP   (INT s0 n0) (INT s1 n1)) = FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))
    f2Tof1 (FLOAT s0 (ZeroPoint n0 (TimesTenToPow s1 n1))) = FL (INT s0 n0) (INT s1 n1)

\end{code}
All the class instances for the user-readable numbers are just converted to the internal float,
operated on, and converted to the external float.
\begin{code}

instance (FloatMarshall t (FLOAT a b), IsZero t r
    ) => IsZero (FLOAT a b) r

instance (FloatMarshall t0 (FLOAT a b), FloatMarshall t1 (FLOAT c d), Compare t0 t1 r  
    ) => Compare  (FLOAT a b) (FLOAT c d) r

instance (FloatMarshall t0 (FLOAT a b), FloatMarshall t1 (FLOAT c d), Sum t0 t1 q, FloatMarshall q r
    ) => Sum      (FLOAT a b) (FLOAT c d) r

instance (FloatMarshall t0 (FLOAT a b), FloatMarshall t1 (FLOAT c d), Subtract t0 t1 q, FloatMarshall q r 
    ) => Subtract (FLOAT a b) (FLOAT c d) r

instance (FloatMarshall t0 (FLOAT a b), FloatMarshall t1 (FLOAT c d), MultD t0 t1 q, FloatMarshall q r 
    ) => MultD    (FLOAT a b) (FLOAT c d) r

\end{code}
Comparison is simple: if the signs differ, compare the signs. If they are the same, compare the numbers.
For floats, compare the bases if the exponents are the same.
Otherwise, make them the same by shifting and incrementing.
\begin{code}
instance                      Compare (INT (:-) n0) (INT (:+) n1) LT
instance                      Compare (INT (:+) n0) (INT (:-) n1) GT

instance (Compare n0 n1 r) => Compare (INT (:+) n0) (INT (:+) n1) r
instance (Compare n1 n0 r) => Compare (INT (:-) n0) (INT (:-) n1) r


\end{code}
Just like addition, we have to shift the base and increment the exponent until both numbers
have the same exponent.
\begin{code}
class ChoseCompare c n n0 n1 n2 | c n n0 n1 -> n2
instance (Compare n0 n n2) => ChoseCompare GT n n0 n1 n2
instance (Compare n n1 n2) => ChoseCompare LT n n0 n1 n2

instance (Compare base0 base1 r) => Compare (FL base0 exp ) (FL base1 exp ) r
instance (Compare exp0  exp1 r0, 
          MkEq r0 (FL base0 exp0) (FL base1 exp1) t0, -- r ~ (t0, (FL base0 exp0), (FL base1 exp1))
          ChoseCompare r0 t0 (FL base0 exp0) (FL base1 exp1) r
    ) => Compare (FL base0 exp0) (FL base1 exp1) r



\end{code}
Addition instances for integers.
If the signs are the same, simply add the natural numbers. 
If they are different, compare the magnitudes of the numbers. If the number to subtract from is greater, we can proceed with the subtraction; 
the sign will not change. If the number to subtract from is smaller, swap the numbers and subtract, and inverse the sign.  
\begin{code}
instance (Sum n0 n1 n2                      ) => Sum (INT s    n0) (INT s    n1) (INT s n2) 
instance (Sum (INT (:+) n1) (INT (:-) n0) r)  => Sum (INT (:-) n0) (INT (:+) n1) r
instance (Compare n0 n1 c, SafeSum c n0 n1 r) => Sum (INT (:+) n0) (INT (:-) n1) r

\end{code}
\hask{SafeSum} assumes a is the positive number and b is the negative number, and we are to perform the subtraction a-b.
\begin{code}
class SafeSum c a b d | c a b -> d
instance SafeSum EQ a b (INT (:+) (D0 :$ T))
instance (InvAdd b a c) => SafeSum LT a b (INT (:-) c)
instance (InvAdd a b c) => SafeSum GT a b (INT (:+) c)

class EqualizeListsR c a b | c a -> b
instance (MkPadding c (a,b) (p0,p1), TAppend a p0 r0, TAppend b p1 r1) => EqualizeListsR c (a,b) (r0,r1)

\end{code}
Addition instances for floats. This class is necessary for avoiding "Incoherent instances" errors.
It checks if the exponents are equal and calls the appropriate class.
\begin{code}
instance (n0 ~ (FL base0 exp0), n1 ~ (FL base1 exp1),
          Compare exp0 exp1 c,
          TypeEq c EQ ce ,
          If ce {-then-} (SumSameExp n0) n1 {-else-} (SumDiffExp n0) n1 {-out-} result
    ) => Sum (FL base0 exp0) (FL base1 exp1) result

\end{code}
Class for adding two floats with the same exponent. 
Ensure the bases are equal in length. Find their sum.
If the sum has more digits than the summands, increment the exponent by one.
\begin{code}
class SumSameExp a b c | a b -> c
instance (EqualizeListsR D0 (base0, base1) (b0e, b1e),
          Sum (INT s0 b0e) (INT s1 b1e) r, r ~ (INT r_s (r0 :$ rr)) ,
          CompLength (r0 :$ rr) b0e c , -- true if lengths are equal, otherwise false
          If c {-then-} Id exp {-else-} Succ exp {-out-} r_exp, 
          exp ~ (INT se ne)
    ) => SumSameExp (FL (INT s0 base0) (INT se ne) ) (FL (INT s1 base1) (INT se ne) ) (FL (INT r_s (r0 :$ rr)) r_exp)

class CompLength a b c | a b -> c
instance CompLength T T TTrue
instance CompLength (a :$ b) T TFalse
instance CompLength T (a :$ b) TFalse
instance (CompLength xs ys r) => CompLength (x :$ xs) (y :$ ys) r


\end{code}
Class for adding two floats with different exponents.
First we identify the larger exponent. We ensure they have the same length. The greater exponent becomes the 'target'.
We pad a zero to the left, decreasing the value of the base; the lesser exponent is increase by one. 
The number that was modified depends on the comparison of the two numbers. 
\begin{code}
class SumDiffExp a b c | a b -> c
instance (Compare (INT se0 exp0) (INT se1 exp1) c,
          EqualizeLists D0 (exp0, exp1) (exp0e, exp1e) ,
          MkEq c (FL (INT s0 base0) (INT se0 exp0e) ) (FL (INT s1 base1) (INT se1 exp1e) ) (t0,t1),
          SumSameExp t0 t1 r
    ) => SumDiffExp (FL (INT s0 base0) (INT se0 exp0) ) (FL (INT s1 base1) (INT se1 exp1)) r

class MkEq c a b d | c a b -> d
--instance EqSum EQ a b d  -- this should never be reached
instance (MkEq GT (FL c d) (FL a b) r) => MkEq LT (FL a b) (FL c d) r
instance (PushAndInc exp0 (FL base1 exp1) c) => MkEq GT (FL base0 exp0) (FL base1 exp1) (FL base0 exp0, c)

class PushAndInc target num r | target num -> r
instance PushAndInc n (FL (INT s base) n) (FL (INT s base) n)
instance (Succ n1 n2, PushAndInc n0 (FL (INT s (D0 :$ base)) n2) r) => PushAndInc n0 (FL (INT s base) n1) r


\end{code}
Instances for subtraction, simply defined as \hask{a - b = a + (-b)}
\begin{code}
instance (n ~ (Negate s1), Sum (INT s0 b0) (INT n b1) r) => Subtract (INT s0 b0) (INT s1 b1) r
instance (n ~ (Negate s1), Sum (FL (INT s0 b0) exp0) (FL (INT n b1) exp1) r) 
  => Subtract (FL (INT s0 b0) exp0) (FL (INT s1 b1) exp1) r


\end{code}
Instances for multiplication;
for integers, we multiply the natural numbers, and the signs;
for floats, we add the exponents and multiply the bases.
However we must first normalize the floats so that multiplication need only handle one case.
The normal form is exactly one leading zero.
\begin{code}
instance MultD (:+) (:+) (:+) 
instance MultD (:-) (:+) (:-)
instance MultD (:+) (:-) (:-)
instance MultD (:-) (:-) (:+)
instance (MultD s0 s1 s2, MultD n0 n1 n2) => MultD (INT s0 n0) (INT s1 n1) (INT s2 n2)

class NormalizeFloatH a b | a -> b
instance (
          TypeEq b0 D0 cb,
          If cb {-then-} Pred exp0 {-else-} Id exp0 {-out-} expn,
          If cb {-then-} Id b0x {-else-} Id (b0 :$ b0x) {-out-} bn,
          If cb {-then-} NormalizeFloatH (FL (INT s0 bn) expn) {-else-} Id (FL (INT s0 bn) expn){-out-} r
    ) => NormalizeFloatH (FL (INT s0 (b0 :$ b0x)) exp0) r

class NormalizeFloat a b | a -> b
instance (
          NormalizeFloatH (FL (INT s0 b0) (INT s1 b1)) (FL (INT s2 b2) exp0), 
          Succ exp0 exp1
    ) => NormalizeFloat (FL (INT s0 b0) (INT s1 b1)) (FL (INT s2 (D0 :$ b2)) exp1)
instance (TypeEq b0 D0 TFalse) => NormalizeFloat (FL (INT s0 (D0 :$ b0 :$ bs)) (INT s1 b1)) (FL (INT s0 (D0 :$ b0 :$ bs)) (INT s1 b1)) 

instance (FloatMarshall q (FLOAT a b), NormalizeFloat q r, FloatMarshall r s) => NormalizeFloat (FLOAT a b) s

instance (
          NormalizeFloat (FL base0' exp0') (FL base0 exp0), 
          NormalizeFloat (FL base1' exp1') (FL base1 exp1), 
          Sum exp0 exp1 exp2', MultD base0 base1 base2, Pred exp2' exp2
    ) => MultD (FL base0' exp0') (FL base1' exp1') (FL base2 exp2)


\end{code}
Instances for testing if a number is zero or is one.
\begin{code}
instance (IsZero n r ) => IsZero (INT s n) r
instance (IsZero n0 r) => IsZero (FL (INT s0 n0) (INT s1 n1)) r

class IsOne z b | z -> b
instance IsOne D0 TFalse
instance IsOne D1 TTrue
instance IsOne D2 TFalse
instance IsOne D3 TFalse
instance IsOne D4 TFalse
instance IsOne D5 TFalse
instance IsOne D6 TFalse
instance IsOne D7 TFalse
instance IsOne D8 TFalse
instance IsOne D9 TFalse
instance (IsOne z r) => IsOne (z  :$ T) r
instance (Nat (x :$ xs), Nat xs, IsZero x r0, IsOne xs r1,
    (r0 :&& r1) r) => IsOne (x :$ xs) r

class DigCount xs n b | xs n -> b
instance DigCount T n n
instance (Pred n n1, DigCount xs n1 b) => DigCount (x :$ xs) n b

instance (IsOne b0 r) => IsOne (INT s0 b0) r
instance (IsOne b0 r0, DigCount b0 b1 z
    ) => IsOne (FL (INT s0 b0) (INT s1 b1)) z

\end{code}
Classes for more readable constraints. 
\begin{code}
class (GenericNumber a) => IntT a
instance (GenericNumber (INT s0 b0), Sign s0, Nat b0) => IntT (INT s0 b0)

class (GenericNumber a) => FloatT a
instance (GenericNumber (FL (INT s0 b0) (INT s1 b1)) ,Sign s0, Sign s1, Nat b0, Nat b1) => 
  FloatT (FL (INT s0 b0) (INT s1 b1))
instance (GenericNumber (FLOAT s0 (ZeroPoint b0 (TimesTenToPow s1 b1))) ,Sign s0, Sign s1, Nat b0, Nat b1) => 
  FloatT (FLOAT s0 (ZeroPoint b0 (TimesTenToPow s1 b1)))

class (GenericNumber a) => Positive a
instance (GenericNumber (NAT a)) => Positive (NAT a)
instance (GenericNumber (INT (:+) b0)) => Positive (INT (:+) b0)
instance (GenericNumber (FL (INT (:+) b0) (INT s1 b1))) => Positive (FL (INT (:+) b0) (INT s1 b1))
instance (GenericNumber (FLOAT Pos (ZeroPoint b0 (TimesTenToPow s1 b1))) , Sign s1, Nat b0, Nat b1) => 
  Positive (FLOAT Pos (ZeroPoint b0 (TimesTenToPow s1 b1)))

class (GenericNumber a) => Negative a
instance (GenericNumber (INT (:-) b0)) => Negative (INT (:-) b0)
instance (GenericNumber (FL (INT (:-) b0) (INT s1 b1))) => Negative (FL (INT (:-) b0) (INT s1 b1))
instance (GenericNumber (FLOAT Neg (ZeroPoint b0 (TimesTenToPow s1 b1))) , Sign s1, Nat b0, Nat b1) => 
  Negative (FLOAT Neg (ZeroPoint b0 (TimesTenToPow s1 b1)))



\end{code}
The successor function. 
\begin{code}
class SuccDig a b | a -> b
instance SuccDig D0 (TTrue, D1)
instance SuccDig D1 (TTrue, D2)
instance SuccDig D2 (TTrue, D3)
instance SuccDig D3 (TTrue, D4)
instance SuccDig D4 (TTrue, D5)
instance SuccDig D5 (TTrue, D6)
instance SuccDig D6 (TTrue, D7)
instance SuccDig D7 (TTrue, D8)
instance SuccDig D8 (TTrue, D9)
instance SuccDig D9 (TFalse, D0)

class SuccW a c | a -> c
instance (SuccDig x (a,b)
    ) => SuccW (x :$ T) (a, b :$ T)
instance (SuccW xs (az,ys),
          If az {-then-} Id (az,x) {-else-} SuccDig x {-out-} (q, xn)
    ) => SuccW (x :$ xs) (q, xn :$ ys)

class Succ a b | a -> b
instance (SuccW a (z,b),
          If z {-then-} Id b {-else-} (TAppend (D1 :$ T)) b {-out-} b2
    ) => Succ a b2

instance (Succ b c) => Succ (INT (:+) b) (INT (:+) c)
instance (IsZero b bz,
          If bz {-then-} Id (D1 :$ T) {-else-} Pred b {-out-} c,
          If bz {-then-} Id (:+) {-else-} Id (:-) {-out-} s
    ) => Succ (INT (:-) b) (INT s c)

    
\end{code}
The predecessor function. 
\begin{code}
class PredDig a b | a -> b
instance PredDig D0 (TFalse, D9)
instance PredDig D1 (TTrue, D0)
instance PredDig D2 (TTrue, D1)
instance PredDig D3 (TTrue, D2)
instance PredDig D4 (TTrue, D3)
instance PredDig D5 (TTrue, D4)
instance PredDig D6 (TTrue, D5)
instance PredDig D7 (TTrue, D6)
instance PredDig D8 (TTrue, D7)
instance PredDig D9 (TTrue, D8)

class PredW a c | a -> c
instance (PredDig x (a,b)
    ) => PredW (x :$ T) (a, b :$ T)
instance (PredW xs (az,ys),
          If az {-then-} Id (az,x) {-else-} PredDig x {-out-} (q, xn)
    ) => PredW (x :$ xs) (q, xn :$ ys)

class Pred a b | a -> b
instance (Nat a, NonZero a, PredW a (TTrue,b)) => Pred a b

instance (Succ b c) => Pred (INT (:-) b) (INT (:-) c)
instance (IsZero b bz,
          If bz {-then-} Id (D1 :$ T) {-else-} Pred b {-out-} c,
          If bz {-then-} Id (:-) {-else-} Id (:+) {-out-} s
    ) => Pred (INT (:+) b) (INT s c)


\end{code}
Computing the length of a list.
\begin{code}
class TLength xs l | xs -> l
instance (TLengthH xs (D0 :$ T) l) => TLength xs l

class TLengthH xs n l | xs n -> l
instance TLengthH T n n
instance (Succ n m, TLengthH xs m r) => TLengthH (x :$ xs) n r


\end{code}
\hask{ToFloat} lifts \hask{INT}s and \hask{NAT}s into \hask{FLOAT}.
\begin{code}
class ToFloat a b | a -> b
instance (Nat (x :$ xs), TLength (x :$ xs) l) => ToFloat (x :$ xs) (FLOAT Pos (ZeroPoint (x :$ xs) (TimesTenToPow Pos l))) 
instance (Nat (x :$ xs), TLength (x :$ xs) l) => ToFloat (INT s (x :$ xs)) (FLOAT s (ZeroPoint (x :$ xs) (TimesTenToPow Pos l))) 
instance ToFloat (FLOAT a (ZeroPoint b (TimesTenToPow c d))) (FLOAT a (ZeroPoint b (TimesTenToPow c d)))


class ToList a where
    toList :: a -> [Float]
instance ToList T where
    toList _ = []
instance (GenericNumber a, ToList b) => ToList (a :$ b) where
    toList _ = (numToFloating (undefined :: a)) : (toList (undefined :: b))

class ToList2 a b where
    toList2 :: a -> [b]
instance ToList2 T Float where
    toList2 _ = []
instance (GenericNumber a, FloatT a, ToList2 b Float) => ToList2 (a :$ b) Float where
    toList2 _ = (numToFloating (undefined :: a)) : (toList2 (undefined :: b))

\end{code}
