Currently contains the following:

Cubic Hermite splines
    usage: plotDataCubicSpline min max stepSize (forces, moments)
    
\begin{code}

module Interpolation where

import System.Random
import System.Environment
import Data.List hiding (groupBy)
import Control.Concurrent (threadDelay, forkIO)
import Control.Concurrent.MVar
import Data.Time.Clock
import System.IO.Unsafe
import Data.Array
import Data.Ix
import Data.Maybe
import System.Process


\end{code}

The below code is gnuplot helper functions. 
\begin{code}

--plotPoints :: C a => [(a, a)] -> IO ()
--plotPoints = plotListStyle [] (PlotStyle Points (CustomStyle [LineType 1, LineWidth 2.0, PointType 7, PointSize 0.2]))
--
--       
--plotFuncs a b c  = GP.plotDefault $ names2d a b c 
--    where names2d domain range x =
--               let frameOpts =
--                      Opts.xRange2d domain $
--                      Opts.yRange2d range $
--                      Opts.deflt
--               in Frame.cons frameOpts $
--                  foldMap (\f ->
--                     fmap (Graph2D.lineSpec
--                              (
--                               LineSpec.deflt)) $
--                     Plot2D.function Graph2D.lines (linearScale 500 domain) f) $ x

\end{code}

Cubic hermite splines in one dimension, using $f(x)$ and $f'(x)$ at $x=x_n, x_n+1$. 
|plotDataCubicSpline| takes a domain, a step size, and the data in the form of a tuple of lists, 
where the first list is the forces and the second list is the slopes. 
\begin{code}

plotDataCubicSpline
  :: (Enum a, Ord a, Floating a, Show a) =>
     a -> a -> a -> ([a], [a]) -> IO ()
plotDataCubicSpline min max step dat = let xVals = [min, min + (step/20) .. max] in do
    writeFile "curve.gp" "set term pngcairo\nset output \"curve.png\"\nplot \"curve0.csv\" using 1:2 with lines"
    writeFile "curve0.csv" $ concat (zipWith (\x y -> show x ++ ", " ++ show y ++"\n") 
                                             xVals 
                                             (map (genPWSpline dat (min, max) step) xVals) 
                                     )
    runCommand "gnuplot curve.gp"
    putStrLn "Show png?" >> getLine >>= 
        (\z -> if z then (>> return ()) (runCommand "curve.png") else return ()) . (=="y")

-- GPs.plotFunc [] [min, min + (step/50) .. max] (genPWSpline dat (min, max) step)

getSplines ((a:b:x), (c:d:y), (e:f:dydx)) = (cubicHermiteSpline a b c d e f):(getSplines ((b:x), (d:y), (f:dydx)))
getSplines _ = []

genPWSpline
      :: (Enum t, Floating t, Ord t, Show t) =>
         ([t], [t]) -> (t, t) -> t -> 
         (t -> t)
genPWSpline dat (xMin, xMax) step = foldFuncs (domPairs domain) splines
        where domain = [xMin, xMin + step .. xMax]
              splines = getSplines (domain, fst dat, snd dat) 
              domPairs (a:b:xs) = (a,b):domPairs (b:xs)
              domPairs _ = []  


\end{code}

foldFuncs takes a list of domains of the type |(Int, Int)| and a list of functions. 
It applies the function to an arbitrary variable if it is in the domain, 
otherwise it recurses. If bottom is reached, it returns |Inf|, 
which gnuplot omits while plotting. 
\begin{code}


foldFuncs (d:[]) (f:[]) x = if x `inD` d then f x else (1.0 / 0.0)
foldFuncs (d:ds) (f:fs) x = if x `inD` d then f x else foldFuncs ds fs x
foldFuncs _ _ _ = (1.0 / 0.0)
inD x (a,b) = x > a && x <= b



\end{code}
Generate a cubic spline between two point, $x_0$ and $x_1$, $using f(x_0), f(x_1), f'(x_0), f'(x_1)$.
The function generated is in absolute coordinates (not relative to $x_0$).
\begin{code}

cubicHermiteSpline x0 x1 y0 y1 m0 m1 x = 
    (h00 (t x))*y0 + 
    (h10 (t x))*(x1 - x0)*m0 +
    (h01 (t x))*y1 + 
    (h11 (t x))*(x1 - x0)*m1
        where h00 t = 2*t^^3 - 3*t^^2 + 1
              h10 t = t^^3 - 2*t^^2 + t
              h01 t = (-2)*t^3 + 3*t^2
              h11 t = t^^3 - t^^2
              t x = (x - x0)/(x1 - x0)
              (^^) = (**)


\end{code}
Function for testing and demonstration purposes which give test data for splines.
\begin{code}

samplePts :: ([Float], [Float])
samplePts = ([32, 40,75,22 ,108, 97 , 156 , 10 , 67 , 203 , 130 , 150 , 34 , 75 , 23 , 84 , 99],
             [10,-13,3 ,-54,7  , 45 , 23  , 23 , 13 , 23  , 43  , 53  , 75 , 84 , 14 ,46  ,-12])

randPts :: (Num a , Random a) => IO ([a], [a])
randPts =  do 
    (a,b) <- randomIO >>= return . split . mkStdGen
    return (randomRs (0,50) a, randomRs (-20,20) b)

randList :: (Num a , Random a) => (a,a) -> IO [a]
randList p = randomIO >>= return . mkStdGen >>= return . randomRs p


-- length: 20
-- elemes: 60
-- stifness: 1e5
---- ghc -prof -auto-all -o Spline Interpolation.lhs -main-is Interpolation.plotSample
--plotSample = plotDataCubicSpline 0.0 200.0 (1/3) (unsafePerformIO randPts :: ([Float], [Float]))
plotSample = getArgs >>= return . head >>= \n -> plotDataCubicSpline 0.0 (read n) 1.0 (unsafePerformIO randPts :: ([Float], [Float]))


sampleData :: ([Float], [Float])
sampleData = 
    ([0.0,1.6759259257888345e-2,6.629938271062395e-2,0.14752777776570214,0.25937037034913557,0.4007716049054575,0.5706944443977184,
      0.7681203703074873,0.9920493826348518,1.2414999998984186,1.5155092591353108,1.8131327159011759,2.1334444442701734,2.475537036834992,
      2.838521604706832,3.2215277775154023,3.623703703408936,4.044216049054182,4.4822499996364105,4.937009258859419,5.407716048945498,
      5.893611110635482,6.393953703188732,6.908021604383109,7.435111110514997,7.974537036399333,8.525632715369508,9.087749999277479,
      9.660259258493673,10.242549381907043,10.834027776925113,11.434120369473849,12.042271603997769,12.657944443459897,13.280620369341761,
      13.909799381643417,14.544999998883403,15.185759258098777,15.831632714845105,16.48219444319644,17.137037035745415,17.795771603603136,
      18.45802777639924,19.123453702281893,19.791716047917795,20.462499998492177,21.135509257708772,21.81046604778982,22.48711110947606,
      23.165203702026762,23.844521603219683,24.524861109351058,25.206037035235724,25.887882714206974,26.570249998116648,27.25300925733505,
      27.936049380751037,28.619277775772005,29.302620368323836,29.98602160285097,30.66944444231636] 
    ,[0.0, 9.999999999181802e-2,0.19669444442834372,0.2901388888651315,0.38038888885773864,0.4674999999617181,0.5515277777326308,0.6325277777260285,
      0.7105555554974627,0.785666666602493,0.8579166665966812,0.927361111035583,0.9940555554747609,1.0580555554697586,1.1194166665761243,
      1.1781944443494097,1.2344444443451668,1.288222222118957,1.3395833332263267,1.3885833332228414,1.4352777776640613,1.4797222221055422,
      1.5219722221028689,1.56208333321158,1.6001111109872606,1.6361111109854216,1.6701388887616253,1.7022499998713787,1.732499999870249,
      1.7609444443137965,1.7876388887575665,1.8126388887570977,1.8359999998679406,1.857777777645601,1.8780277776456487,1.8968055554235579,
      1.9141666665348707,1.930166666535118,1.9448611109798257,1.9583055554245532,1.970555555424851,1.981666666536288,1.9916944443144229,
      2.0006944443148407,2.008722222093203,2.015833333204963,2.0220833332056984,2.027527777650966,2.0322222220961663,2.036222222096891,
      2.039583333208671,2.042361110987049,2.044611110987588,2.0463888887658603,2.047749999877311,2.048749999877528,2.049444444322124,
      2.0498888887665867,2.0501388887666057,2.050249999877658,2.0502777776554364])
\end{code}



\begin{code}



--bilinearInterpolation :: Array (Int, Int) (Maybe Float) -> Int -> Array (Int, Int) (Maybe Float)
testArr = array ((0,0),(1,1)) [ ((0,0),Just 2.0), ((0,1),Just 3.0), ((1,0),Just 1.0), ((1,1),Just 4.0)]
testArr1 (x,y) = array ((0,0),(x,y)) $ zip [(a,b) | a <- [0..x] , b <- [0..y]] $ ( (map Just . unsafePerformIO) (randList (0,5.0)) :: [Maybe Float] ) 

testArr2 :: IO [ Array (Int, Int) (Maybe Float) ]
testArr2 = readFile "test.txt" >>= return . read
 

bilinearInterpolation :: (Integral a, RealFrac b, Ix a, Enum b) =>
     a -> Array (a, a) (Maybe b) -> Array (a, a) (Maybe b)
     
bilinearInterpolation step arr =  accumArray accFunc Nothing newDomain $ map toIntIx $ concatMap (toFunc . (toGroup arr)) (assocs arr)
    where toIntIx ((x,y),z) = ((round $ x/newStep, round $ y/newStep), z)
            
          toFunc (Just ((x',y'), q0, q1, q2, q3)) = zip pts $ map (Just . (bilinearF x (x+1) y (y+1) q0 q1 q2 q3)) pts
            where (x,y) = (fromIntegral x', fromIntegral y')
                  pts = [ (a,b) | a <- [x, x+newStep .. x+(1-(newStep))], b <- [y, y+newStep .. y+(1-(newStep))] ]
          toFunc Nothing = []
         
          newStep = 1/(fromIntegral step)
          ((xMin', yMin'), (xMax', yMax')) = bounds arr
          [xMin, xMax, yMin, yMax] = map ((\x -> round (x / newStep)) . fromIntegral)  [xMin', xMax', yMin', yMax']
          newDomain = ((xMin,yMin), (xMax, yMax))
          
toGroup arr ((x,y),z) 
            | not (inArr arr (x+1,y) && 
               inArr arr (x,y+1) &&  
               inArr arr (x+1,y+1) ) = Nothing
               
            | (isJust (arr!(x+1,y)) && 
               isJust (arr!(x,y+1)) && 
               isJust (arr!(x+1,y+1)) && 
               isJust z )
                        = Just $ ((x,y), fromJust z, 
                                         fromJust (arr!(x,y+1)), 
                                         fromJust (arr!(x+1,y)), 
                                         fromJust (arr!(x+1,y+1)) )
            | otherwise = Nothing
            
inArr arr i = isJust $ find (==i) (indices arr)
         
accFunc :: Maybe a -> Maybe a -> Maybe a
accFunc Nothing q = q
accFunc q Nothing = q
accFunc r q       = q

bilinearF x1 x2 y1 y2 q11 q12 q21 q22 (x,y) = (*) (1/((x2-x1)*(y2-y1))) $
    q11*(x2-x)*(y2-y) + q21*(x-x1)*(y2-y) + q12*(x2-x)*(y-y1) + q22*(x-x1)*(y-y1)




{- experimental implementation -}
bilinear2 step arr = groupBy $ map groupBy toList
    where toList = [[arr!(x,y) | x <- [xMin .. xMax]] | y <- [yMin .. yMax]]
          ((xMin, yMin), (xMax, yMax)) = bounds arr
          getFunc (a,b,c,d) = bilinearF 0 1 0 1 a b c d
          groupBy' ([a,b],[c,d]) = [tup a b, tup c d]
          groupBy' _             = []
          
tup (a,b) (c,d) = (a,b,c,d)

groupBy (a:b:xs) = (a,b):(groupBy (b:xs))
groupBy (a:xs)   = []

f = [[a+b | a <- [1..3]] | b <- [1..3]]

\end{code} 





