This module includes some examples of the application of type level safety to the finite element method.
\begin{code}
{-# LANGUAGE 
    FunctionalDependencies,
    FlexibleInstances,
    ScopedTypeVariables,
    UndecidableInstances,
    TypeFamilies ,
    FlexibleContexts,
    ConstraintKinds,
    GADTs,
    RankNTypes,
    ImpredicativeTypes,
    IncoherentInstances,
    TypeOperators
    #-}
module Prof where


import UnitDefinitions
import Data.List
import Data.Array.Unboxed
import Data.Array.MArray
import Data.Array.IO
import Data.Maybe
import Interpolation

                               
--stiffness matrix for an element with 2DOF at each node:
--translation and rotation
getStiffnessMatrix :: (Floating v) => v -> v -> Int -> [((Int, Int), v)]
getStiffnessMatrix l ei i = zip [(a,b)|a <- [i..i+3], b <- [i..i+3]] m
    where m = map (*(ei/l**3))  [12,  6*l,   -12,   6*l   , 
                                6*l,  4*l^2, -6*l,  2*l^2 , 
                                -12,  -6*l,  12,    -6*l  ,
                                6*l,  2*l^2, -6*l,  4*l^2 ]
                                
                                

--creates global matrix, gets the matrix for each element, then creates an array, adding
--elements when multiple entires occupy the same space
getGlobalStiffnessMatrix :: (Floating v, IArray UArray v) => v -> Int -> v -> UArray (Int, Int) v
getGlobalStiffnessMatrix l num ei = accumArray (+) 0 ((1,1),(num*2+2,num*2+2)) $ concatMap elemMatrix [1..num] 
    where lElem = l/(fromIntegral num) --length of each element
          elemMatrix n = getStiffnessMatrix lElem ei (n*2-1) --bounds of matrix moved over by 2 for each element

--gets an initial value vector from the problem parameters
initVector :: (Floating v, IArray UArray v) =>  v -> Int -> [v] -> [v] -> UArray Int v
initVector l num forces moments 
    | num /= (length forces) || num /= (length moments) = error "bad array length"
    | otherwise = listArray (1, num*2+2) $ concat ( zipWith (\a b -> [a,b]) forces moments ) ++ dupEnd
        where dupEnd = map (head . reverse) [forces, moments]
                                            
                                            
-- applies boundary conditions as well as combining the matrices
applyBoundaryCond :: (Floating v, IArray UArray v) => UArray Int v -> UArray (Int, Int) v -> BoundaryCondition -> UArray (Int, Int) v
applyBoundaryCond initVector globalKM (SetToZero bc) = listArray  --create array
                                                        (shiftBounds $ bounds augmentedMatrix) --with these bounds
                                                        (map (\((_,_),z) -> z) $ --remove (x,y) values
                                                             filter (\((x,y),z) -> not (elem x bc || elem y bc)) --filter rows and cols in the bc
                                                                    (assocs augmentedMatrix)) 
    where shiftBounds ((a,b),(c,d)) = ((a,b),(c-(length bc),d-(length bc))) --shrinks bounds by # of items in BC
          augmentedMatrix = combine globalKM initVector 

--places the vector in a column at the far right of the global stiffness matrix
--this will cause an error if the length of the vector is too short
combine :: (Floating v, IArray UArray v) => UArray (Int, Int) v -> UArray Int v -> UArray (Int, Int) v 
combine matrix vector = array (fullBounds (bounds matrix)) $ assocs matrix ++ assocs expand --creates a new array from the 2 arrays
    where expand = ixmap (newBounds (bounds vector) (bounds matrix)) (\(x,y) -> y) vector --upgrades the vector to a 2d array with width 1
          newBounds (a,b) ((_,_),(c,_))= ((c+1,a), (c+1,b))
          fullBounds ((a,b),(c,d)) = ((a,b),(c+1,d))
     

gaussElimination ::
     UArray (Int, Int) Float -> IO (UArray (Int, Int) Float)
gaussElimination matrix' = do
    matrix <- (thaw matrix'  :: IO (IOUArray (Int, Int) Float ) )
    ((x1, y1) , (x2, y2)) <- getBounds matrix
    
    
    {- not adding so many zeroes here could really speed this up -}
    mapM_
        (\c -> let elimRow c rs = 
                         mapM_ (\d -> do
                                   headVal <- readArray matrix (c,d)
                                   curRow <- getRow d matrix
                                   addRowConst d (zipWith (+) curRow (map (*(-headVal)) rs) ) matrix
                                   ) [c+1 .. y2] in do       
            pivotVal <- readArray matrix (c,c)
            case pivotVal of
                0.0 -> error "pivot is 0"
                1.0 -> getRow c matrix >>= elimRow c
                v   -> (divRow c v matrix) >> getRow c matrix >>= elimRow c
        ) [y1..y2]
    
    
    mapM_ (\d -> do {- the row and column of the leading one -}
        {- we know right away the values of interest and we avoid adding zeroes -}
        pivRow <- mapM (\c -> readArray matrix (c,d) -- get the non zero values in the pivot row
                         ) [d..x2] 
        mapM_
            (\c -> do {- all the numbers above it -}
                aboveP <- readArray matrix (d,c) {- the # above the pivot ; add its negative to create a zero -}
--                addRow c (-aboveP) d matrix
                mapM_ (\(e,v) -> do {- e, the column to add to currently (row is c); v, the value of pivot row in this column -}
                        cur <- readArray matrix (e,c) 
                        writeArray matrix (e,c) (cur - aboveP*v)
                      ) (zip [d .. x2] pivRow)
               
            ) [y1 .. d - 1]
           ) [y1 + 1 .. y2]
           
    showM matrix ----- remove later
    freeze matrix :: IO (UArray (Int, Int) Float )


solve len num lForce lMoment boundC stiffness = do
    solveM <- gaussElimination $ applyBoundaryCond --solves matrix with given parameters
                                        (initVector len num lForce lMoment)
                                        (getGlobalStiffnessMatrix len num stiffness)
                                        boundC
    return $ toTuple $ replaceBC boundC (results solveM)
    where replaceBC (SetToZero (l:[])) list = (take (l-1) list) ++ [0] ++ (drop (l-1) list) --inserts boundary conditions back into solution vector 
          replaceBC (SetToZero (l:ls)) list = (take (l-1) next) ++ [0] ++ (drop (l-1) next)
            where next = (take (l-1) list) ++ [0] ++ (drop (l-1) list)  
          targetCol m = ((fst . snd) (bounds m)) --column containing desired results
          results m = map (\((_,y),a) -> a) $ filter (\((x,_),_) -> x==(targetCol m)) (assocs m) --filter desired results from matrix and clean list
          toTuple list = (map (\(_,y) -> y) $ fst split, map (\(_,y) -> y) $ snd split)
            where split = partition (\(x,_) -> odd x) $ zip [1..] list --first list is displacement, second is rotation

plotSolve len num lForce lMoment boundC stiffness = do
    solution <- solve len num lForce lMoment boundC stiffness 
    plotDataCubicSpline 0.0 len (len / fromIntegral num) solution
    return solution
    
-- returns a list corresponding to the row
getRow r m = do
    ((x1, y1) , (x2, y2)) <- getBounds m
    mapM (\c -> readArray m (c,r)
         ) [x1..x2]
         
-- performs r1(i) = vals(i) forall i
addRowConst r1 vals m = do
    mapM_ 
        (\(c,v) -> 
            writeArray m (c,r1) v
        ) (zip [1..] vals)

-- performs r1 = r1 + n*r2
addRow r1 n r2 m = do
    ((x1, y1) , (x2, y2)) <- getBounds m
    mapM_ (\c -> do
        r1v <- readArray m (c,r1) 
        r2v <- readArray m (c,r2)
        writeArray m (c,r1) (r1v + n*r2v) 
            ) [x1 .. x2]
            
-- perform r1 = r1 / n
divRow r1 n m = do
    ((x1, y1) , (x2, y2)) <- getBounds m
    mapM_ (\c -> do
        val <- readArray m (c,r1) 
        writeArray m (c,r1) (val/n)
            ) [x1 .. x2]

{- compile with
    ghc -prof -auto-all -o Prof Prof.lhs -main-is Prof.main
   run with
    Prof +RTS -p
 -}
main = plotSolve 20.0 100 (map (0.1*) [0 .. 99]) (replicate 100 (0.0)) (twoFixedEnd 100) 10000.0
--main = solve' 20.0 60 (replicate 60 (5.0)) (replicate 60 (0.0)) fixedEndFreeEnd 10000.0
--main = solveM
--    where len = 20.0; num = 60; lForce = replicate 60 (5.0); lMoment = replicate 60 (5.0); stiffness = 10000;
--          solveM = gaussElimination' $ applyBoundaryCond --solves matrix with given parameters
--                                        (initVector len num lForce lMoment)
--                                        (getGlobalStiffnessMatrix len num stiffness)
--                                        fixedEndFreeEnd


--    
--showM :: IOUArray (Int, Int) Float -> [()]
showM (arr :: IOUArray (Int, Int) Float) = do
     ((x1, y1) , (x2, y2)) <- getBounds arr
     sequence_ $ concat $ map (++[putStr "\n\n"]) $ ((map . map) (\i -> (readArray arr i >>= putStr . (++" ") . show)) [[(a,b) | a <- [x1..x2]] | b <- [y1..y2] ])
     print ""
--     
        
--        
          
-- SetToZero: contains the 1-centered index of element that is zero
-- should be sorted in ascending order
data BoundaryCondition = SetToZero [Int] deriving Show

fixedEndFreeEnd = SetToZero [1,2] --one fixed end
twoFixedEnd num = SetToZero [1,2,num*2+1,num*2+2] --both ends fixed- takes # elemenets
fixedEndRotatingEnd num = SetToZero [1,2,num*2+1] 
freeEndProppedEnd num = SetToZero [1,2,num*2+2] --one end fixed, one end fixed rotation
fixedEndsFixedMiddle num 
    | even num = SetToZero [1,2,num,num+1,num*2+1,num*2+2] --fixed at 3 points
    | odd num  = SetToZero [1,2,num+1,num+2,num*2+1,num*2+2]


\end{code}