\ignore{
\begin{code}

{-# LANGUAGE EmptyDataDecls, MultiParamTypeClasses, FunctionalDependencies, FlexibleInstances, RankNTypes, ScopedTypeVariables, UndecidableInstances #-}
module SIUnits where

import UnitDefinitions
import Prelude hiding ((+), negate, (-), (*),(/))
import qualified Prelude ((+), negate, (-), (*),(/))

\end{code}
}


\section{SI Units}\label{sec:SIUnits}

This module defines a value as a number and a unit, and defines valid operations on values, as well as the addition of units.




The data structure for values, along with type aliases for several common units.
\begin{code}
data SIUnitVal u1 u2 u3 u4 u5 u6 val = SIUnitVal val
type Meter val = SIUnitVal M1 Kg0 S0 A0 Mol0 K0 val
type PerM val = SIUnitVal M_1 Kg0 S0 A0 Mol0 K0 val
type MPerS val = SIUnitVal M1 Kg0 S_1 A0 Mol0 K0 val
type Second val = SIUnitVal M0 Kg0 S1 A0 Mol0 K0 val
type Kilogram val = SIUnitVal M0 Kg1 S0 A0 Mol0 K0 val
type Joule val  = SIUnitVal M2 Kg1 S_2 A0 Mol0 K0 val

type Unitless val = SIUnitVal M0 Kg0 S0 A0 Mol0 K0 val

\end{code}
Instances of \hask{Show} for units which gives the \LaTeX{} code for displaying the units.
\begin{code}

instance Show M_5   where show _ = "m^{-5}"
instance Show M_4   where show _ = "m^{-4}"
instance Show M_3   where show _ = "m^{-3}"
instance Show M_2   where show _ = "m^{-2}"
instance Show M_1   where show _ = "m^{-1}"
instance Show M0    where show _ = ""
instance Show M1    where show _ = "m"
instance Show M2    where show _ = "m^{2}"
instance Show M3    where show _ = "m^{3}"
instance Show M4    where show _ = "m^{4}"
instance Show M5    where show _ = "m^{5}"
instance Show Kg_5  where show _ = "kg^{-5}"
instance Show Kg_4  where show _ = "kg^{-4}"
instance Show Kg_3  where show _ = "kg^{-3}"
instance Show Kg_2  where show _ = "kg^{-2}"
instance Show Kg_1  where show _ = "kg^{-1}"
instance Show Kg0   where show _ = ""
instance Show Kg1   where show _ = "kg"
instance Show Kg2   where show _ = "kg^{2}"
instance Show Kg3   where show _ = "kg^{3}"
instance Show Kg4   where show _ = "kg^{4}"
instance Show Kg5   where show _ = "kg^{5}"
instance Show Mol_5 where show _ = "mol^{-5}"
instance Show Mol_4 where show _ = "mol^{-4}"
instance Show Mol_3 where show _ = "mol^{-3}"
instance Show Mol_2 where show _ = "mol^{-2}"
instance Show Mol_1 where show _ = "mol^{-1}"
instance Show Mol0  where show _ = ""
instance Show Mol1  where show _ = "mol"
instance Show Mol2  where show _ = "mol^{2}"
instance Show Mol3  where show _ = "mol^{3}"
instance Show Mol4  where show _ = "mol^{4}"
instance Show Mol5  where show _ = "mol^{5}"
instance Show S_5   where show _ = "s^{-5}"
instance Show S_4   where show _ = "s^{-4}"
instance Show S_3   where show _ = "s^{-3}"
instance Show S_2   where show _ = "s^{-2}"
instance Show S_1   where show _ = "s^{-1}"
instance Show S0    where show _ = ""
instance Show S1    where show _ = "s"
instance Show S2    where show _ = "s^{2}"
instance Show S3    where show _ = "s^{3}"
instance Show S4    where show _ = "s^{4}"
instance Show S5    where show _ = "s^{5}"
instance Show A_5   where show _ = "A^{-5}"
instance Show A_4   where show _ = "A^{-4}"
instance Show A_3   where show _ = "A^{-3}"
instance Show A_2   where show _ = "A^{-2}"
instance Show A_1   where show _ = "A^{-1}"
instance Show A0    where show _ = ""
instance Show A1    where show _ = "A"
instance Show A2    where show _ = "A^{2}"
instance Show A3    where show _ = "A^{3}"
instance Show A4    where show _ = "A^{4}"
instance Show A5    where show _ = "A^{5}"
instance Show K_5   where show _ = "K^{-5}"
instance Show K_4   where show _ = "K^{-4}"
instance Show K_3   where show _ = "K^{-3}"
instance Show K_2   where show _ = "K^{-2}"
instance Show K_1   where show _ = "K^{-1}"
instance Show K0    where show _ = ""
instance Show K1    where show _ = "K"
instance Show K2    where show _ = "K^{2}"
instance Show K3    where show _ = "K^{3}"
instance Show K4    where show _ = "K^{4}"
instance Show K5    where show _ = "K^{5}"

\end{code}

The class which indicates the ordering of SI base units in the \hask{SIUnitVal} datatype. 
It also defines an instance of \hask{Show} for a value with a unit, which shows the value and the \LaTeX{} for the entire unit.
\begin{code}

class SIUnit a where

instance (UnitM m, Show m, UnitKg kg, Show kg, UnitS s, Show s, UnitA a, Show a, UnitMol mol, Show mol, UnitK k, Show k, Show val) 
    => SIUnit (SIUnitVal m kg s a mol k val) where

instance (UnitM m, Show m, UnitKg kg, Show kg, UnitS s, Show s, UnitA a, Show a, UnitMol mol, Show mol, UnitK k, Show k, Show val) 
    => Show (SIUnitVal m kg s a mol k val) where
        show (SIUnitVal val) = unwords $ filter (not . null) [show val, show m, show kg,  show s, show a, show mol, show k]
            where
              m   = undefined :: m
              kg  = undefined :: kg
              a   = undefined :: a
              mol = undefined :: mol
              s   = undefined :: s
              k   = undefined :: k

\end{code}


Classes defining valid multiplication and addition over values with units. 
\begin{code}

class Mult a b c | a b -> c, a c -> b, b c -> a where
  (*) :: a -> b -> c

class Div a b c | a b -> c, a c -> b, b c -> a where
  (/) :: a -> b -> c

class Add a where
  (+) :: a -> a -> a
  negate :: a -> a
  (-) :: a -> a -> a
\end{code}

Instances of \hask{Mult} and \hask{Add} for basic numeric types: \hask{Double} and \hask{(Num a) => [a]}. 
Note that there is no instance of \hask{Mult} for a list; currently we cannot guarantee that the lists (which are used to represent vectors) would have the same length. 
\begin{code}
instance Mult () () () where 
    () * () = ()

instance Div () () () where 
    () / () = ()

instance Add () where 
    () + () = ()
    negate () = ()
    () - () = ()

instance Mult Double Double Double where 
    a * b = a Prelude.* b
    
instance Div Double Double Double where 
    a / b = a Prelude./ b
    
instance Add Double where 
    a + b = a Prelude.+ b 
    negate a = Prelude.negate a 
    a - b = a Prelude.- b

zipWith' f [] ys = ys
zipWith' f xs [] = xs
zipWith' f (x:xs) (y:ys) = (f x y):(zipWith' f xs ys)

instance (Prelude.Num a) => Add [a] where  
    xs + ys = zipWith' (Prelude.+) xs ys
    xs - ys = zipWith' (Prelude.-) xs ys
    negate = map Prelude.negate

\end{code}

Instances of \hask{Add} and \hask{Mult} for values with units. 
Addition is only defined for values with identical units.
Multiplication defines the units of the resultant type through \hask{UnitDefinitions.AddUnit}.
\begin{code}

instance (UnitM u1, UnitKg u2, UnitS u3, UnitA u4, UnitMol u5, UnitK u6, Add val) 
    => Add (SIUnitVal u1 u2 u3 u4 u5 u6 val) where
  (SIUnitVal x) + (SIUnitVal y) = SIUnitVal (x + y)
  negate (SIUnitVal x) = SIUnitVal (negate x)
  (SIUnitVal x) - (SIUnitVal y) = SIUnitVal (x - y)  
  
instance (AddUnit u1 v1 w1, AddUnit u2 v2 w2, AddUnit u3 v3 w3, AddUnit u4 v4 w4, AddUnit u5 v5 w5, AddUnit u6 v6 w6, Mult val val val) 
    => Mult (SIUnitVal u1 u2 u3 u4 u5 u6 val) (SIUnitVal v1 v2 v3 v4 v5 v6 val) (SIUnitVal w1 w2 w3 w4 w5 w6 val) where
  (SIUnitVal a) * (SIUnitVal b) = SIUnitVal (a * b)

-- a/b = c   =>   a = bc
instance (AddUnit v1 w1 u1, AddUnit v2 w2 u2 , AddUnit v3 w3 u3, AddUnit v4 w4 u4, AddUnit v5 w5 u5, AddUnit v6 w6 u6, Div val val val) 
    => Div (SIUnitVal u1 u2 u3 u4 u5 u6 val) (SIUnitVal v1 v2 v3 v4 v5 v6 val) (SIUnitVal w1 w2 w3 w4 w5 w6 val) where
  (SIUnitVal a) / (SIUnitVal b) = SIUnitVal (a / b)

\end{code}


It is convenient to define \hask{Add} and \hask{Mult} classes for tuples of unit types,
pointwise, in order to facilitate arithmetic with fractional type-level numbers. 
\begin{code}
instance (Add a, Add b) => Add (a,b) where
  (a,b) + (c,d) = (a+c,b+d)
  (a,b) - (c,d) = (a-c,b-d)
  negate (a,b) = (negate a, negate b)

instance (Mult a1 b1 c1, Mult a2 b2 c2) => Mult (a1,a2) (b1,b2) (c1,c2) where
  (a1,a2) * (b1,b2) = (a1*b1, a2*b2)

instance (Div a1 b1 c1, Div a2 b2 c2) => Div (a1,a2) (b1,b2) (c1,c2) where
  (a1,a2) / (b1,b2) = (a1/b1, a2/b2)

\end{code}
