This module contains code that tests the consistency of type level multiplication, non-zero multiplication, and addition. 

\begin{code}

module TestCode where 

import UnitDefinitions
import NumericTypes
import Examples

import Data.List
import System.Command
import System.Random

\end{code}
This function will produce multiple Haskell source files containing sample multiplications of random numbers, 
compiles the files in parallel, and run each compiled binary consecutively. 
It takes two arguments: the number of files to produce, and the number of multiplications to perform.
\begin{code}

--todo: sometimes parrallel compiles output to the stream simultaneously created a bunch of garbled text.
--get each stream and handle it line by line, prefixing the creator of each line?
testAsync nCases nProcs = do
    putStrLn "Writing source files..."
    writeRand nCases nProcs
    putStrLn "Compiling..."
    h <- mapM compileFile [0..(nProcs - 1)]
    mapM_ waitForProcess h
    sequence_ ( map (\x -> readProcess ("test\\testcase" ++ alpha x ++ ".exe") [] [] >>=
                           putStrLn . (++) ("\nRunning test case " ++ alpha x ++ "\n")                                
                    ) [0..(nProcs - 1)] )
\end{code}

-- (\x -> readProcess ("test\\testcase" ++ alpha x ++ ".exe") [] [] >>= (\y -> putStrLn ("Running test case " ++ alpha x ++ "\n" ++ y)))

This function produces multiple Haskell source files for checking type level multiplication. It takes two arguements: the number of files to produce, 
and the number of multiplications per file.
\begin{code}
writeRand nCases nProcs = do 
    g <- randomIO
    let r = rands (nProcs*nCases) g 10
    sequence_ $ zipWith (\xs n -> testNonZeroMultToFile xs 10 n) (splitInto nCases r) [0..7]
\end{code}



Helper functions.
\begin{itemize}
    \item |compileFile| takes the index of the file, and compiles that file, returning the |ProcessHandle| corresponding to the created instance of GHC.
    \item |splitInto| takes a list and splits it into sublists of length |l|.
    \item |rands| takes three integers, $n, g, k$ , and produces a list of random numbers whose length is $n$, whose generator is $g$, which is bounded by $(0,10^k)$. The random numbers are weighted by the logistic curve. 
    \item |alpha| translates an |Int| to a string, where 0 is |'A'| and 25 is |'Z'|.
\end{itemize}
\begin{code}
compileFile s = putStrLn (cmd) >> createProcess (shell cmd) >>= \(_,_,_,h) -> return h 
    where fname = "test\\testcase" ++ alpha s ++ "."
          cmd   = "ghc --make -w -o " ++ fname ++ "exe " ++ fname ++ "hs"

splitInto _ [] = []
splitInto l xs 
    | length xs >= l = [take l xs] ++ splitInto l (drop l xs)
    | otherwise      = [xs]

rands n g k = take n $ filter f (zip a b) 
    where a        = map t $ randomRs (0 :: Float, 9) ga  
          b        = map t $ randomRs (0 :: Float, 9) gb  
          (ga, gb) = split $ mkStdGen g
          d        = 2.5
          f        = (\(x,y) -> x*y < (10^k) && x*y >0 )
          t        = (round . (\x -> 10^k*(1/(1+e**((-x+6)*d))))) 
          e        = exp 1
          
alpha :: Int  -> String
alpha =  (\x -> [x]) . toEnum . (+) 65

\end{code}

This function creates a string representing a Haskell source file which will perform type level multiplication of non-zero numbers and compare 
the result against the actual value of the multiplication, printing a failure message if needed; if no multiplications fail, a single success message is printed at the end.
The function takes two arguements: a list of tuples representing multiplicands, and the width of type level number to multiply. Currently multiplication is only implemented
for width two, three, and ten type-level numbers; there is no error checking for numbers of other width and attempting to compile a file generated with other widths will generate
compile-time errors.  
\begin{code}
testNonZeroMult list k = (intercalate "\n" (map concat
 [["{-# LANGUAGE ScopedTypeVariables #-}"] 
 ,["module Main where"] 
 ,["import SizeTypes"]  
 ,["f :: forall e f g . (MultDNonZero e f g, NonZero f, NonZero e, NonZero g, MultD e f g, Size e, Size f, Size g) => (e, f, Int) -> IO Bool"] 
 ,["f (a, b, s)"]
 ,["    | s == g = putStr \"\" >> return True"]
 ,["    | otherwise = putStrLn (\"Mult of \" ++ show (toInt a) ++ \" and \" ++ show (toInt b) ++ \" failed\") >> return False"]
 ,["        where g = toInt (u :: g) "] 
 ,["u = undefined"] 
 ,["main = putStrLn \"Testing type-level multiplication of " , 
    show (length list), " numbers.\" >> ",
    "((\\b -> if and b then putStr \"All mults passed\" else putStr \"\") =<< sequence"] 
 ,[" ["] 
 ,[intercalate ",\n"  (map makeF list)]
 ,[" ])"]    
 ]))
    where makeF (a,b) = " f (u :: SIZE" ++ show0 k ++ " " ++ showAsD a ++ 
                        ", u :: SIZE" ++ show0 k ++ " " ++ showAsD b ++ 
                        "," ++ show (a*b) ++ ")" 
          showAsD n = concatMap (\x -> 'D':x:" ") $ replicate (k - length (show n)) '0' ++ show n
          show0 n 
            | n == 10 = ""  
            | otherwise = show n

\end{code}

This function writes the file to the folder "\test" with name "testcase#.hs" where "#" is a character from 'A' to 'Z'. 
\begin{code}
testNonZeroMultToFile l k num = writeFile ("test\\testcase" ++ alpha num ++ ".hs") (testNonZeroMult l k)
\end{code}

These two functions are synonymous with the above functions, instead taking as arguements a lower and upper bound and writing instances for each multiplication involving all numbers
in the intersection of the given range, and the bounds of the type level number of given width.
\begin{code}
testNZMrange a b k = testNonZeroMult [(a,b)| a <- [(max 0 a)..(min (10^k - 1) b)], b <- [(max 0 a)..(min (10^k - 1) b)], a*b < 10^k] k
testNZMrangeToFile a b k num = writeFile ("test\\testcase" ++ alpha num ++ ".hs") (testNZMrange a b k)


\end{code}
 --ghc --make -o test\testcasea.exe test\testcasea.hs
 
