\begin{code}
{-# LANGUAGE TemplateHaskell, FlexibleInstances, UndecidableInstances, QuasiQuotes, RankNTypes #-}

module TypeGen 
    (mkDecWith, nat, int, float, num, mkTestFunc, mkFuncListRand, mkFuncsMany, mkFuncList, mkFuncList2, 
     mkFuncListM, mkFuncListM2, compareDefinition, add2Definition, timesDefinition, sub2Definition,
     mkUnitDataDecs, mkUnitClasses, mkUnitAddClasses)
        where

import Language.Haskell.TH
import Data.Char
import Data.Word
import Data.List
import Data.List.Split
import Numeric
import Control.Monad
import Language.Haskell.TH.Quote
import System.Random

--import NumericTypes
--import TypeLevelList
--import FloatingPoint

import System.IO.Unsafe
import Language.Haskell.Syntax

\end{code}
Helper functions.
\begin{code}

mkDecWith a b = mkDecWith' a b >>= return . (:[])
mkDecWith' name exp = exp >>= \expr -> (funD (mkName name) [(return $ Clause [] (NormalB expr) [])])
mkDecWith2 name expr = FunD (mkName name) [Clause [] (NormalB expr) []]

trimChar c = reverse . dropWhile (==c) . reverse . dropWhile (==c)
trimCharBy f = reverse . dropWhile f . reverse . dropWhile f
trimChars [] str = str
trimChars (x:xs) str = trimChars xs (trimChar x str)
sgnNum n = if (head n) == '-' then "Neg" else "Pos"
absNum n = if (head n) == '-' then tail n else n
noSp = trimChar ' '

foldQ con app [] = (con . mkName) "End"
foldQ con app (x:xs) = app (app list x) (foldQ con app xs)
      where list = (con . mkName) ":$"

\end{code}
Functions which allow creating of declerations, types, and expressions from different types.
\begin{code}

mkNatGeneric:: (Name -> a) -> (a -> a -> a) -> String -> Q a
mkNatGeneric con app num 
  | not $ and $ map (`elem` "0123456789 ") num = error $ "Not a valid natural: " ++ num
  | not $ and $ map (`elem` "0123456789") noSp = error $ "Non-leading or -trailing spaces found: " ++ num
  | otherwise = return (fold digNames)
        where digStrs = (map (("D" ++) . (:[])) noSp) ++ ["End"]
              digNames = map (con . mkName) digStrs
              fold (x:[]) = x
              fold (x:xs) = app (app list x) (fold xs)
                    where list = (con . mkName) ":$"
              noSp = reverse $ dropWhile (==' ') $ reverse $ dropWhile (==' ') num

mkNatType = mkNatGeneric ConT AppT
mkNatExp  = mkNatGeneric ConE AppE


mkIntGeneric :: (String -> Q a) -> (Name -> a) -> (a -> a -> a) -> String -> Q a
mkIntGeneric mkNatF con app num 
    | not $ and $ map (`elem` "0123456789- ") num = error "Not a valid integer"
    | not $ and $ map (`elem` "0123456789-") (noSp num) = error "Non-leading or -trailing spaces found."
    | otherwise = mkNatF (absNum (noSp num)) >>= return . app (app (con (mkName "INT")) (con (mkName (sgnNum (noSp num))))) 

mkIntType = mkIntGeneric mkNatType ConT AppT
mkIntExp  = mkIntGeneric mkNatExp  ConE AppE

mkFloatGeneric :: (String -> Q a) -> (Name -> a) -> (a -> a -> a) -> String -> Q a
mkFloatGeneric mkNatF con app str 
    | not $ and $ map (`elem` "0123456789.-e ") str = error $ "Invalid character found while parsing " ++ str
    | not $ and $ map (`elem` "0123456789.-e") (noSp str) = error $ "Non-leading or -trailing spaces found while parsing " ++ str
    | otherwise =  unify (parseFloat b, (read e :: Int))
        where check [base]           = if not (null base) then (base, "0") else error $ "Base or exponent are empty: " ++ str
              check [base, exponent] = if not (null base) && not (null exponent) then (base, exponent) else error $ "Base or exponent are empty: " ++ str
              check _                = error $ "Invalid number ('e' present more than once): " ++ str
              (b,e) = check $ splitOn "e" (noSp str)    
              -- (FL s0 (ZeroPoint n0 (TimesTenToPow s1 n1)))
              unify ((m, e0), e1) = do
                vMant <- mkNatF mant
                vExp  <- mkNatF exp
                return $ app2 fl sMant (app2 zp vMant (app2 tp sExp vExp))
                  where e = e1 + e0
                        sMant = con $ mkName $ sgnNum m
                        sExp  = con $ mkName $ sgnNum $ show e
                        mant = absNum m
                        exp  = show $ abs e
                        fl = con $ mkName "FLOAT"
                        zp = con $ mkName "ZeroPoint"
                        tp = con $ mkName "TimesTenToPow"
                        app2 f x y = app (app f x) y

parseFloat str 
      | or $ map (=='-') (tail str) = error $ "Invalid number (negative not at beginning of number) : "  ++ str
      | (<) 1 $ length $ filter (=='.') str = error $ "Invalid number (multiple decimal points found) : "  ++ str
      | otherwise = (base, exponent)
          where absStr = if (head str) == '-' then tail str else str
                pStr = if (head absStr) == '.' then '0':absStr else absStr
                exponent = length $ takeWhile (/='.') pStr
                base = (sign ++ (filter (/= '.') pStr))
                sign = if (head str) == '-' then "-" else ""

mkFloatType = mkFloatGeneric mkNatType ConT AppT
mkFloatExp  = mkFloatGeneric mkNatExp  ConE AppE


\end{code}
Makes a tuple in the appropriate context.
\begin{code}
mkTuple :: (Name -> a) -> (a -> a -> a) -> [Q a] -> Q a
mkTuple con app nums' = do
    nums@(x:xs) <- sequence nums'
    return nums
    let tup = con $ mkName $ "(" ++ replicate ((length nums) - 1) ',' ++ ")"
    return $ mkTuple' tup nums
        where mkTuple' tup (x:[]) = app tup x
              mkTuple' tup (x:xs) = mkTuple' (app tup x) xs

\end{code}
Generic quoter which will use the most specific number possible.
\begin{code}

mkNumericGeneric :: (String -> Q a) -> (Name -> a) -> (a -> a -> a) -> String -> Q a
mkNumericGeneric r con app str 
    | '/' `elem` str = case splitOn "/" str of 
        [a,b] -> mkFloatGeneric r con app $ show $ (fromIntegral (read a :: Integer)) / (fromIntegral (read b :: Integer))
        _ -> error $ "Invalid construction of rational: " ++ str
    | (head str') == '(' && (last str') == ')' = mkTuple con app $ map (mkNumericGeneric r con app) (splitOn "," ( (tail . init) str' ))   
    | not $ or $ map (`elem` str) "e-.+" = mkNatGeneric con app str
    | '+' == (head (dropWhile (==' ') str)) = mkIntGeneric r con app (tail (dropWhile (==' ') str))
    | not ('e' `elem` str || '.' `elem` str) = mkIntGeneric r con app str
    | otherwise = mkFloatGeneric r con app str
        where str' = noSp str

mkNumericType = mkNumericGeneric mkNatType ConT AppT
mkNumericExp  = mkNumericGeneric mkNatExp  ConE AppE


\end{code}
Quoter which generates vectors.
\begin{code}
mkVectorGeneric :: (String -> Q a) -> (Name -> a) -> (a -> a -> a) -> [Char] -> Q a
mkVectorGeneric r con app str = do
    nNums <- mapM (mkNumericGeneric r con app) nums
    return $ foldQ con app nNums
        where nums = splitOn "," str

mkVectorType = mkVectorGeneric mkNatType ConT AppT
mkVectorExp  = mkVectorGeneric mkNatExp  ConE AppE


\end{code}
Quoter which generates matrices.
\begin{code}
mkMatrixGeneric :: (String -> Q a) -> (Name -> a) -> (a -> a -> a) -> [Char] -> Q a
mkMatrixGeneric r con app str = do
    nNums <- (mapM . mapM) (mkNumericGeneric r con app) nums
    return $ foldQ con app (map (foldQ con app) nNums)
        where nums 
                | '[' `elem` str = trimCharBy (\(x:_) -> x == "") $ doSplit "],[" "[]"
                | '\r' `elem` str = trimCharBy (\(x:_) -> x == "") $ (doSplit "\r\n" "\r\n")
                    where doSplit c d = (map . map) (filter (not . (`elem` d))) $ map (splitOn ",") $ splitOn c (filter (/=' ') str)

mkMatrixType = mkMatrixGeneric mkNatType ConT AppT
mkMatrixExp  = mkMatrixGeneric mkNatExp  ConE AppE
test = "[[1,3,5,7,4,5],[2,3,5,3,2,3],[4,2,3,5,3]]"


\end{code}
The quoters.
\begin{code}
-- QuasiQuoter {quoteExp, quotePat, quoteType, quoteDec}
qq a b = QuasiQuoter a undefined b undefined
float = qq mkFloatExp   mkFloatType
int   = qq mkIntExp     mkIntType   
nat   = qq mkNatExp     mkNatType 
num   = qq mkNumericExp mkNumericType 
mat   = qq mkMatrixExp  mkMatrixType
list  = qq mkVectorExp  mkVectorType



\end{code}
Given a class of the form:
    class Class t0 t1 ... tn | t0 t1 .. t(n-1) -> tn
Generate a function of the form:
    tClass :: Class t0 t1 .. tn => t0 -> t1 -> ... -> tn
    tClass = undefined
\begin{code}
mkClassFunc name = do
    (ClassI (ClassD _ cName typeVars _ _) _) <- reify name 
    let tlist = map (VarT . getTyVarName) typeVars
        newName = (mkName (fname cName))
        {- forall <type vars> . Class <typevars> => <t0> -> <t1> -> ... -> <tn> -}
        typeSig = ForallT typeVars [ClassP name tlist] (foldType tlist) 
        in return [
            {- tClass :: forall <type vars> . Class <typevars> => <t0> -> <t1> -> ... -> <tn> -}
            SigD newName typeSig  ,
            {- tClass = undefined -}
            FunD newName [Clause [] (NormalB (VarE (mkName "undefined"))) []]  ]
                where fname n | (head tname) == 'T' = 't':(tail tname) | otherwise = 't':tname
                        where tname = last (splitOn "." (show n))
                      getTyVarName (PlainTV n) = n
                      getTyVarName (KindedTV n _) = n -- is this ever called? 
                      foldType (x:[]) = x
                      foldType (x:xs) = AppT (AppT ArrowT x) (foldType xs)


\end{code}
The code below generates one digit arithmetic classes.
Generics.
\begin{code}
sequence' xs = sequence xs >>= return . concat

threeTVclass n = return [ ClassD [] (mkName n) (map PlainTV l) [FunDep [a,b] [c]] [] ]
    where l@[a,b,c] = map mkName ["a","b","c"]

fourTVclass n  = return [ ClassD [] (mkName n) (map PlainTV l) [FunDep [a,b] [c,d]] [] ]
    where l@[a,b,c,d] = map mkName ["a","b","c","d"]


--arithInstances :: (Int -> Int -> Int) -> String -> Q [Dec]
arith4Instances name f = return $
    [InstanceD [] (foldl AppT (ConT (mkName name)) (types a b)) []     | a <- [0..9], b <- [0..9]    ]
        where types a b = map (ConT . mkName) ["D" ++ show a, "D" ++ show b, 'D':hi:[] , 'D':lo:[] ]
                  where (lo:hi:_) = reverse $ "0" ++ (show (a `f` b))

\end{code}
Add2 class.
\begin{code}
add2Definition = sequence' [add2Class, add2Instances]

add2Class :: Q [Dec]
add2Class = fourTVclass "Add2"

add2Instances :: Q [Dec]
add2Instances = arith4Instances "Add2" (+)

\end{code}
Times class.
\begin{code}
timesDefinition = sequence' [timesClass, timesInstances]

timesClass :: Q [Dec]
timesClass = fourTVclass "Times"

timesInstances :: Q [Dec]
timesInstances = arith4Instances "Times" (*)


\end{code}
Subtraction class.
\begin{code}
sub2Definition = sequence' [sub2Class, sub2Instances]

sub2Class :: Q [Dec]
sub2Class = threeTVclass "Sub2"

sub2Instances :: Q [Dec]
sub2Instances = return $
    [InstanceD [] (foldl AppT (ConT (mkName "Sub2")) (types a b)) []     | a <- [0..9], b <- [0..9] ]
        where types a b = map (ConT . mkName) ["D" ++ show a, "D" ++ show b, "D" ++ (show ((10 + a - b) `rem` 10)) ]

\end{code}
Comparison class.
\begin{code}
compareDefinition = sequence' [compareClass, compareInstances]

compareClass :: Q [Dec]
compareClass = threeTVclass "Compare"

compareInstances :: Q [Dec]
compareInstances = return $
    [InstanceD [] (foldl AppT (ConT (mkName "Compare")) (types a b)) []     | a <- [0..9], b <- [0..9]    ]
        where types a b = map (ConT . mkName) ["D" ++ show a, "D" ++ show b, show (compare a b) ]

\end{code}
The code below is for testing type level numbers.
\begin{code}
data Op = Sum | Mult | Sub | Comp deriving (Enum, Show)
typeFunc Sum   = "tSum"
typeFunc Mult  = "tMultD"
typeFunc Sub   = "tSubtract"
typeFunc Comp  = "tCompare"

dataFunc Sum   = "+"
dataFunc Mult  = "*"
dataFunc Sub   = "-"
dataFunc Comp  = "compare"

valueFunc Sum  = "numToIntegral"
valueFunc Mult = "numToIntegral"
valueFunc Sub  = "numToIntegral"
valueFunc Comp = "toDataComp"

\end{code}
Given n0, n1, op, produces an expression of the form
(valueFunc op) (mkNatExp n0 `typeFunc op` mkNatExp n1) == (n0 `dataFunc op` n1)
\begin{code}
mkTestFunc :: Word32 -> Word32 -> Op -> Q Exp
mkTestFunc n1 n0 op = typeComp >>= \t -> return $  AppE (AppE eq dataComp) (AppE valF t) --UInfixE eq dataComp (AppE valF t) --
    where [m0,m1] = map (mkNatExp . show) [n0,n1] 
          valF = VarE $ mkName $ valueFunc op
          eq = VarE $ mkName "=="
          dataComp = AppE (AppE (VarE $ mkName $ dataFunc op) q0) q1
          typeComp = do 
            m0' <- m0
            m1' <- m1
            return $ AppE (AppE (VarE $ mkName $ typeFunc op) m0') m1'
          [q0,q1] = map (LitE . IntegerL . toInteger) [n0,n1]

mkFuncList :: String -> ([Word32], [Word32]) -> Op -> Q [Dec]
mkFuncList str (n0,n1) op = do
    funcs <- sequence $ zipWith3 mkTestFunc n0 n1 (repeat op)
    decs <- return $ zipWith mkDecWith2 names funcs
    test <- mkDecWith str $ return $ ListE names2-- fold list t names
    return (decs ++ test)
        where names2 = map (VarE . mkName) names
              names = take l $ map (("f"++) . show) [0..]
              l = length n0 

mkFuncList2 :: String -> [(Word32, Word32)] -> Op -> Q [Dec]
mkFuncList2 str nums op = mkFuncList str (pack nums) op

mkFuncListM :: String -> IO ([Word32], [Word32]) -> Op -> Q [Dec]
mkFuncListM str nums op = runIO nums >>= \z -> mkFuncList str z op

mkFuncListM2 :: String -> IO [(Word32, Word32)] -> Op -> Q [Dec]
mkFuncListM2 str nums op = runIO nums >>= \z -> mkFuncList str (pack z) op


randList :: (Num a , Random a) => (a,a) -> IO [a]
randList p = randomIO >>= return . mkStdGen >>= return . randomRs p

randList2 :: (Num a , Random a) => (a,a) -> IO ([a], [a])
randList2 q = randomIO >>= return . mkStdGen >>= return . f q
    where f (a,b) gen0 = (q : qs, r : rs)
            where (q, gen1) = randomR (a,b) gen0
                  (r, gen2) = randomR (q,b) gen1
                  (qs,rs) = f (a,b) gen2

randList2' :: (Num a, Random a) => (a, a) -> IO [(a, a)]
randList2' p = randList2 p >>= return . unpack

pack :: [(a, b)] -> ([a], [b])
pack []         = ([],[])
pack ((a,b):xs) = let (ax,bx) = pack xs in (a:ax, b:bx)

unpack :: ([a], [b]) -> [(a, b)]
unpack ([] , [])        = []
unpack ((x:xs), (y:ys)) = (x,y) : (unpack (xs,ys))

mkFuncsMany n range = do
    (n0,n1,ops) <- runIO $ do
                (z,w) <- randList2 range >>= \(xs,ys) -> return (take n xs, take n ys) >>= (print >> return)
                y <- randList (0,3) >>= return . take n
                return (z,w,y)
    zipWithM mkDecWith' names ( zipWith3 mkTestFunc n0 n1 (map toEnum ops) ) 
        where names = map (("f"++) . show) [0..]
              

mkFuncListRand str n range = do
    funcs <- mkFuncsMany n range
    test <- mkDecWith str $ return $ ListE names-- fold list t names
    return (funcs ++ test)
        where names = take n $ map (VarE . mkName . ("f"++) . show) [0..]

genToken []       = return []
genToken ((a:b) : xs) = let x = (toUpper a):b in genToken xs >>= return . (:) ( DataD [] (mkName x) [] [NormalC (mkName x) []] [] ) 

\end{code}
Generate things pertaining to units.
\begin{code}

mkUnitDataDecs :: Q [Dec]
mkUnitDataDecs = return [DataD [] (mkName (showU unit pow))  [] [] [] | unit <- ["M", "Kg", "A", "S", "Mol", "K"], pow <- [-10 .. 10]]
    where showU s n = s ++ p ++ (show (abs n))
            where p = if n < 0 then "_" else ""

mkUnitClasses :: Q [Dec]
mkUnitClasses = return [InstanceD [] (ConT (mkName ("Unit" ++ unit)) `AppT` (showU unit pow)) (mkUnitFuncDec unit pow) | unit <- ["M", "Kg", "A", "S", "Mol", "K"], pow <- [-10 .. 10]]
    where showU s n = ConT $ mkName $ s ++ p ++ (show (abs n))
            where p = if n < 0 then "_" else ""
          mkUnitFuncDec unit pow = [FunD (mkName ("toInt" ++ unit)) [Clause [WildP] (NormalB (LitE (IntegerL pow))) []]]

mkUnitAddClasses :: Q [Dec]
mkUnitAddClasses = return $ map (\h -> InstanceD [] h []) head
        where showU s n = ConT $ mkName $ s ++ p ++ (show (abs n))
                where p = if n < 0 then "_" else ""
              head = [(ConT (mkName "AddUnit")) `AppT` (showU unit a) `AppT` (showU unit b) `AppT` (showU unit (a+b)) | 
                          unit <- ["M", "Kg", "A", "S", "Mol", "K"], a <- [-10 .. 10], b <- [-10 .. 10], (abs (a+b)) < 11]


\end{code}

