/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																						 *
 *	Orbitwise adjacency tree (binary red-black tree implementation) Abstract Data Type.  *
 *																						 *
 *									Source file											 *
 *																						 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include"owe_header.h"

static orbitwise_adjacency_tree_node orbitwise_adjacency_tree_sentinel = { orbitwise_adjacency_tree_NIL, orbitwise_adjacency_tree_NIL, 0, orbitwise_adjacency_tree_BLACK, 0};

void orbitwise_adjacency_tree_initialize(orbitwise_adjacency_tree_ptr tree) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbitwise_adjacency_tree_initialize
	*  -----------------------------------
	*  Parameters:
	*		- Memory address of the red-black orbitwise adjacency tree on which the operation will be performed.
	*		  Pointer to pointer is used to access the information, so in the function call the parameter
	*		  must use the address operator '&' before the name of the variable.
	*
	*		The function initialize the value of the pointer to be the root of an empty
	*		red-black orbitwise adjacency tree.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	tree->root = orbitwise_adjacency_tree_NIL;
	tree->size = 0;
}


void orbitwise_adjacency_tree_delete(orbitwise_adjacency_tree_ptr tree) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbitwise_adjacency_tree_delete
	*  -----------------------------------------
	*  Parameters:
	*		- Memory address of the red-black orbitwise adjacency tree on which the operation will be performed.
	*		  Pointer to pointer is used to access the information, so in the function call the parameter
	*		  must use the address operator '&' before the name of the variable.
	*
	*		The function initialize the value of the pointer to be the root of an empty
	*		red-black orbitwise adjacency tree.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


	orbitwise_adjacency_tree_delete_tree_structure(tree);
	tree->size = 0;
}



coordinate orbitwise_adjacency_tree_insert(orbitwise_adjacency_tree_ptr tree, coordinate canonical_representative, slave_process_orbit_tree_ptr slave_process_slave_process_tree_ptr) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  orbitwise_adjacency_tree_insert
	 *  -----------------------------------------
	 *  Parameters:
	 *		- Memory address of the red-black orbitwise adjacency tree on which the operation will be performed.
	 *		  Pointer to pointer is used to access the information, so in the function call the parameter
	 *		  must use the address operator '&' before the name of the variable.
	 *
	 *		- Coordinate of the canonical representative (inmediatelly followed by the denominator)
	 *		  of the orbit to be inserted into the red-black orbit tree.
	 *  
	 *  Return values:
	 *		- If the coordinate was not previously inserted:
	 *           A new node is created (allocated and initialized) in its corredponsing location of
	 *           the red black orbit tree. its corresponding coordinate is also allocated in memory
	 *           and its value set to a copy of that provided in the coordinate input parameter.
	 *           The adjacency counter of the node is set to 0. Finally, the value returned is a pointer
	 *			 to the newly allocated coordinate.
	 *
	 *      - If the coordinate was previously inserted:
	 *			  The red black orbit tree structure remains intact, no memory is allocated. The
	 *			  adjacency counter of the corresponding orbit is incremented. Finally
	 *            the function returns 0.
	 *
	 *  Note: this function does not modify or de-allocate its input parameter.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbitwise_adjacency_tree_node *current, *parent, *x;


    /* find future parent */
	current = tree->root;
    parent = 0;
    while (current != orbitwise_adjacency_tree_NIL) {
		if (orbitwise_adjacency_tree_compEQ(canonical_representative, current->orbit_canonical_representative)) {
			current->count++;
			return current->orbit_canonical_representative;
		}
		
        parent = current;
		current = orbitwise_adjacency_tree_compLT(canonical_representative, current->orbit_canonical_representative) ?
            current->left : current->right;
    }

	/* Increment the size of the tree. */
	tree->size++;

    /* setup new node */
    if ((x = (orbitwise_adjacency_tree_node*) malloc (sizeof(*x))) == 0) owe_error_handler_raise_error("memory allocation error (orbitwise_adjacency_tree_node variable : orbitwise_adjacency_tree_insert : orbitwise_adjacency_tree ADT)", 1);
    /* set the red-black tree node properties */
	x->parent = parent;
    x->left = orbitwise_adjacency_tree_NIL;
    x->right = orbitwise_adjacency_tree_NIL;
    x->color = orbitwise_adjacency_tree_RED;


	/* Send the potentially new orbit to the slave orbit tree. */
	slave_process_orbit_tree_insert(slave_process_slave_process_tree_ptr, canonical_representative);


	/* Create coordinate for the orbit in the adjacency tree. */
	if (!(x->orbit_canonical_representative = (coordinate) malloc(sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))) owe_error_handler_raise_error("memory allocation error (coordinate variable : orbitwise_adjacency_tree_insert function: orbitwise_adjacency_tree ADT)", 1);
	
	/* set the coordinate value of the newly inserted orbit */
	memcpy((x->orbit_canonical_representative),canonical_representative,sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1));

	x->count=1;
    /* insert node in tree */
    if(parent) {
		if(orbitwise_adjacency_tree_compLT(canonical_representative, parent->orbit_canonical_representative))
            parent->left = x;
        else
            parent->right = x;
    } else {
        tree->root = x;
    }

    orbitwise_adjacency_tree_insertFixup(&tree->root, x);

    return x->orbit_canonical_representative;
}



coordinate orbitwise_adjacency_tree_find(orbitwise_adjacency_tree_ptr tree, coordinate canonical_representative) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbitwise_adjacency_tree_find
	*  -----------------------------
	*  Parameters:
	*		- Memory address of the red-black orbitwise adjacency tree on which the operation will be performed.
	*		  Pointer to pointer is used to access the information, so in the function call the parameter
	*		  must use the address operator '&' before the name of the variable.
	*
	*		- Coordinate of the canonical representative (inmediatelly followed by the denominator)
	*		  of the orbit whose presence in the red-black orbitwise adjacency tree wants to be determined
	*		  (whether it is in the tree or not).
	*
	*  Return values:
	*		- If the red-black orbitwise adjacency tree contains the coordinate:
	*			 The value returned is a pointer to the coordinate in the tree whose value is
	*			 equal to that of the parameter.
	*
    *		- If the red-black orbitwise adjacency tree does not contain the coordinate:
	*			 The returned value is 0.
	*
	*  Note: this function does not modify or de-allocate its input parameter.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


    orbitwise_adjacency_tree_node_ptr current = tree->root;
    while(current != orbitwise_adjacency_tree_NIL) {
        if(orbitwise_adjacency_tree_compEQ(canonical_representative, current->orbit_canonical_representative)) {
            return current->orbit_canonical_representative;
        } else {
            current = orbitwise_adjacency_tree_compLT (canonical_representative, current->orbit_canonical_representative) ?
                current->left : current->right;
        }
    }
    return 0;
}



void orbitwise_adjacency_tree_print_tree(orbitwise_adjacency_tree_ptr tree) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbitwise_adjacency_tree_print_tree
	*  -----------------------------------
	*  Parameters:
	*		- Orbitwise adjacency tree to be printed
	*
	*	This functions prints on the standard output stream the tree passed as a parameter inorder (iterative).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbitwise_adjacency_tree_node_ptr node;
	orbitwise_adjacency_tree_node_ptr *node_stack;
	unsigned long int stack_top = 0;

	node_stack = (orbitwise_adjacency_tree_node_ptr*) malloc(sizeof(orbit_tree_node_ptr)*(2+2*owe_lib_log_base_2(1+tree->size)));

	printf("        Count  Orbit (canonical representative)\n ----------------------------------------------\n"); 

	if(tree->root != orbitwise_adjacency_tree_NIL) {
		node = tree->root->left;
		node_stack[stack_top++] = tree->root;
		while(node != orbitwise_adjacency_tree_NIL || stack_top) {
			if(node != orbitwise_adjacency_tree_NIL) {
				node_stack[stack_top++] = node;
				node = node->left;
			}
			else
			{
				node = node_stack[--stack_top];
				printf(" %c %10ld  ", (!memcmp(node->orbit_canonical_representative,owe_setting.metric_polytope.cut_canonical_representative,sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))?'C':((!memcmp(node->orbit_canonical_representative,owe_setting.metric_polytope.anticut_canonical_representative,sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))?'A':'N'), node->count);
				orbit_print_coordinate(node->orbit_canonical_representative);printf("\n");
				node = node->right;
			}

		}

	}

	printf("\n C = cut; A = anticut; N = non-cut/anticut\n");

}


void orbitwise_adjacency_tree_inorder(orbitwise_adjacency_tree_node *node) {

	/**************************
    *  In order tree parsing  *
    **************************/
	if(node!=orbitwise_adjacency_tree_NIL) {
		orbitwise_adjacency_tree_inorder(node->left);
		printf("%12ld  ",node->count);
		orbit_print_coordinate(node->orbit_canonical_representative);printf("\n");
		orbitwise_adjacency_tree_inorder(node->right);
	}
}



static void orbitwise_adjacency_tree_rotateLeft(orbitwise_adjacency_tree_node_ptr *tree, orbitwise_adjacency_tree_node *x) {

   /**************************
    *  rotate node x to left *
    **************************/

    orbitwise_adjacency_tree_node *y = x->right;

    /* establish x->right link */
    x->right = y->left;
    if (y->left != orbitwise_adjacency_tree_NIL) y->left->parent = x;

    /* establish y->parent link */
    if (y != orbitwise_adjacency_tree_NIL) y->parent = x->parent;
    if (x->parent) {
        if (x == x->parent->left)
            x->parent->left = y;
        else
            x->parent->right = y;
    } else {
        *tree = y;
    }

    /* link x and y */
    y->left = x;
    if (x != orbitwise_adjacency_tree_NIL) x->parent = y;

}

static void orbitwise_adjacency_tree_rotateRight(orbitwise_adjacency_tree_node_ptr *tree, orbitwise_adjacency_tree_node *x) {

   /****************************
    *  rotate node x to right  *
    ****************************/

    orbitwise_adjacency_tree_node *y = x->left;

    /* establish x->left link */
    x->left = y->right;
    if (y->right != orbitwise_adjacency_tree_NIL) y->right->parent = x;

    /* establish y->parent link */
    if (y != orbitwise_adjacency_tree_NIL) y->parent = x->parent;
    if (x->parent) {
        if (x == x->parent->right)
            x->parent->right = y;
        else
            x->parent->left = y;
    } else {
        *tree = y;
    }

    /* link x and y */
    y->right = x;
    if (x != orbitwise_adjacency_tree_NIL) x->parent = y;

}

static void orbitwise_adjacency_tree_insertFixup(orbitwise_adjacency_tree_node_ptr *tree, orbitwise_adjacency_tree_node *x) {

   /*************************************
    *  maintain Red-Black tree balance  *
    *  after inserting node x           *
    *************************************/

    /* check Red-Black properties */
    while (x != *tree && x->parent->color == orbitwise_adjacency_tree_RED) {
        /* we have a violation */
        if (x->parent == x->parent->parent->left) {
            orbitwise_adjacency_tree_node *y = x->parent->parent->right;
            if (y->color == orbitwise_adjacency_tree_RED) {

                /* uncle is RED */
                x->parent->color = orbitwise_adjacency_tree_BLACK;
                y->color = orbitwise_adjacency_tree_BLACK;
                x->parent->parent->color = orbitwise_adjacency_tree_RED;
                x = x->parent->parent;
            } else {

                /* uncle is BLACK */
                if (x == x->parent->right) {
                    /* make x a left child */
                    x = x->parent;
                    orbitwise_adjacency_tree_rotateLeft(tree, x);
                }

                /* recolor and rotate */
                x->parent->color = orbitwise_adjacency_tree_BLACK;
                x->parent->parent->color = orbitwise_adjacency_tree_RED;
                orbitwise_adjacency_tree_rotateRight(tree, x->parent->parent);
            }
        } else {

            /* mirror image of above code */
            orbitwise_adjacency_tree_node *y = x->parent->parent->left;
            if (y->color == orbitwise_adjacency_tree_RED) {

                /* uncle is RED */
                x->parent->color = orbitwise_adjacency_tree_BLACK;
                y->color = orbitwise_adjacency_tree_BLACK;
                x->parent->parent->color = orbitwise_adjacency_tree_RED;
                x = x->parent->parent;
            } else {

                /* uncle is BLACK */
                if (x == x->parent->left) {
                    x = x->parent;
                    orbitwise_adjacency_tree_rotateRight(tree, x);
                }
                x->parent->color = orbitwise_adjacency_tree_BLACK;
                x->parent->parent->color = orbitwise_adjacency_tree_RED;
                orbitwise_adjacency_tree_rotateLeft(tree, x->parent->parent);
            }
        }
    }
    (**tree).color = orbitwise_adjacency_tree_BLACK;
}



void orbitwise_adjacency_tree_delete_tree_structure(orbitwise_adjacency_tree_ptr tree) {
	
   /*************************************
    *  Orbit tree structure deletion    *
    *  (iterative tree deletion).       *
    *************************************/

	orbitwise_adjacency_tree_node_ptr node, next_node;
	orbitwise_adjacency_tree_node_ptr *node_stack;
	unsigned long int stack_top = 0;

	node_stack = (orbitwise_adjacency_tree_node_ptr*) malloc(sizeof(orbit_tree_node_ptr)*(2+2*owe_lib_log_base_2(1+tree->size)));

	if(tree->root != orbitwise_adjacency_tree_NIL) {
		node = tree->root->left;
		node_stack[stack_top++] = tree->root;
		while(node != orbitwise_adjacency_tree_NIL || stack_top) {
			if(node != orbitwise_adjacency_tree_NIL) {
				node_stack[stack_top++] = node;
				node = node->left;
			}
			else
			{
				node = node_stack[--stack_top];
				next_node = node->right;
				
				/* Deletion of orbit coordinate. */
				free(node->orbit_canonical_representative);
		
				/* Deletion of node in tree. */
				free(node);
				
				node = next_node;
			}

		}

	}

}

