/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *							  OWE Slave process									   *
 *																				   *
 *								  Source file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "owe_header.h"

/* Slave process orbit tree. */
static slave_process_orbit_tree slave_process_slave_process_tree;

void owe_slave_process_initialize(void) {

	/* CDD Initialization. */
	dd_set_global_constants();

	/* Initialize the slave process orbit tree. */
	slave_process_orbit_tree_initialize(&slave_process_slave_process_tree);

}

void owe_slave_process_finalize(void) {

	/* Finalize metric polytope. */
	metric_polytope_finalize();

	/* Finalize the slave process orbit tree. */
	slave_process_orbit_tree_delete(&slave_process_slave_process_tree);


}



void owe_slave_process_execute(void) {

	/* Initialize the slave process. */
	owe_slave_process_initialize();

	/* Recieve from the master process the broadcast indicating which metric polytope will
	be computed, and initialize the metric polytope ADT and canon library accordinly. */
	owe_slave_process_get_parameter_broadcast();

	/* Compute orbits as they are received. */
	owe_slave_process_process_orbits();
	
	/* Finalize the slave process. */
	owe_slave_process_finalize();

}



void owe_slave_process_process_orbits(void) {

	coordinate orbit_canonical_representative;	/* This variable will contain the canonical representative of the orbits to be computed. */
	MPI_Status status;	/* Variable used to retrieve the value of the MPI function calls. */
	unsigned char orbit_computation_status; /* This variable will contain the value returned by the orbit_processor_process_orbit function, which
											is 1 if the orbit was computed and 0 if the orbit was skipped (due to its adjacency being outside range). */

	/* Allocate memory for the orbit canonical representative. */
	if(!(orbit_canonical_representative = (coordinate) malloc(sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))) owe_error_handler_raise_error("memory allocation error (coordinate variable : owe_slave_process_process_orbits function : orbit_processor ADT)", 1);

	do {
		/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
		/*DEBUG*/ printf("\n\nAwaiting message from master process...");
		/*DEBUG*/ fflush(stdout);
		/*DEBUG*/ #endif

		/* Recieve the MPI message containing either the next orbit to compute, or an indication (through the message tag) that the computation is complete. */
		MPI_Recv(orbit_canonical_representative, owe_setting.metric_polytope.dimension + 1, MPI_UNSIGNED, 0, MPI_ANY_TAG, MPI_COMM_WORLD, &status);

		/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
		/*DEBUG*/ printf("\n\nMessage recieved");
		/*DEBUG*/ fflush(stdout);
		/*DEBUG*/ #endif

		/* If the message tag indicates there is an orbit to be computed, compute it. */
		if(status.MPI_TAG == OWE_MPI_TAG_PROCESS_ORBIT) {
					
			/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
			/*DEBUG*/ printf("\n\nComputing orbit: "); orbit_print_coordinate(orbit_canonical_representative); printf("...\n\n");
			/*DEBUG*/ fflush(stdout);
			/*DEBUG*/ #endif

			/* Perform the orbit computation. */
			orbit_computation_status = orbit_processor_process_orbit(orbit_canonical_representative, &slave_process_slave_process_tree);

			/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
			/*DEBUG*/ if(orbit_computation_status) printf("\n\nFinished computing orbit ");
			/*DEBUG*/ else printf("\n\nSkipped orbit orbit ");
			/*DEBUG*/ orbit_print_coordinate(orbit_canonical_representative); printf("\n\n");
			/*DEBUG*/ fflush(stdout);
			/*DEBUG*/ #endif

			/* Notify master processor orbit computation is complete. */
			MPI_Send(orbit_canonical_representative, 0, MPI_UNSIGNED, 0, (orbit_computation_status)?OWE_MPI_TAG_SLAVE_FINISHED_COMPUTING_ORBIT:OWE_MPI_TAG_SLAVE_SKIPPED_ORBIT, MPI_COMM_WORLD);
		}

	} while (status.MPI_TAG != OWE_MPI_TAG_FINALIZE); /* Finalize when the tag of the message recieved indicates computation is complete. */

	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
	/*DEBUG*/ printf("\n\nFinalizing");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

	/* Free the memory taken by the orbit canonical representative. */
	free(orbit_canonical_representative);

}




void owe_slave_process_get_parameter_broadcast(void) {

	unsigned int parameters[4];	/* Variable used for getting the parameter set from master process broadcast. */

	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
	/*DEBUG*/ printf("\n\nAwaiting for initialization broadcast");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

	/* Receibe the broadcast from the master process indicating which metric polytope will
	be computed and initialize canon and matric polytope ADT accordinly. Also, the adjacency range. */
	MPI_Bcast(&parameters, 4, MPI_UNSIGNED, 0, MPI_COMM_WORLD);

	/* Initialize canon. */
	init_canon(parameters[0]);
			
	/* Initialize metric polytope. */
	metric_polytope_initialize((unsigned char)parameters[0]);

	/* Set the adjacency range. */
	owe_setting.adjacency_lower_bound = parameters[1];
	owe_setting.adjacency_upper_bound = parameters[2];

	/* Set the double description time limit */
	owe_setting.double_description_time_limit = parameters[3];

	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_SLAVE
	/*DEBUG*/ printf("\n\nBroadcast received, process is now ready to start computing orbits");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

}
