%%%  src/INet/Description.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{Net Descriptions}\sectlabel{INet.Description}

\begin{ModuleHead}
\begin{code}
module INet.Description where

import INet.Polarity
import INet.Utils.Vector (Vector)
\end{code}
\end{ModuleHead}

Whereas in \sectref{INet.Polar.INet}, we introduced types for nets
considered as run-time states,
here we introduce \emph{net description}
for static representation of, in particular, rule right-hand sides.

The following types are dictated by our current choice of
array implementation (|Data.Vector| from the |vector| package, for efficiency),
but aliased for readability:
\begin{code}
type PI  = Int -- ``port index''
type NI  = Int -- ``node index''
\end{code}
%
The port index type |PI|
will be used also in actual nets, while
the node index type |NI|
is needed only for right-hand side nodes
in descriptions and during creation.
%we do not have global node arrays at run-time.\linebreak
%
We arbitrarily call the two nodes engaged in an interaction ``source'' and ``target'';
%with our use of polarities,
the ``source'' interface consists of the auxiliary ports of the node with the ``function''
label with negative principal port, and the ``target'' interface
consists of the auxiliary ports of the ``constructor'' node
with positive principal port.
The following data type serves to identify all ports in a rule's right-hand side
(the ``|!|'' specifies strict constructor argument positions
 for efficiency):

\begin{code}
data PortTargetDescription
  =  SourcePort    !PI
  |  InternalPort  !NI !PI  -- node, port
  |  TargetPort    !PI      deriving (Eq, Ord, Show)
\end{code}
%% -- |Show| for debugging, |Eq| for |checkNetDescription|, |Ord| for |ptdRel|
%
Therefore, each RHS node is described by its label
and by the connections of \emph{all} its ports:
%\edcomm{WK}{For |portDescriptions|, strictness may be a problem for |Snapshot|.}

\begin{code}
data NodeDescription nLab = NodeDescription
  { nLab              ::                 ! nLab
  , portDescriptions  :: {-# UNPACK #-}  ! (Vector PortTargetDescription)
  }
\end{code}
%
A |NetDescription| is intended as description of the RHS of interaction rules:
%It contains assumptions about the arities of the two LHS nodes
%implicit in the |bounds| of the |source| and |target| arrays.

\begin{code}
data NetDescription nLab = NetDescription
  { source  :: {-# UNPACK #-} ! (Vector PortTargetDescription)
  , target  :: {-# UNPACK #-} ! (Vector PortTargetDescription)
  , nodes   :: {-# UNPACK #-} ! (Vector (NodeDescription nLab))
  }
\end{code}
%
%%
%% sourceInterval :: NetDescription nLab -> (PI, PI)
%% sourceInterval = bounds . source
%%
%% targetInterval :: NetDescription nLab -> (PI, PI)
%% targetInterval = bounds . target
%%
%% nodeInterval :: NetDescription nLab -> (NI, NI)
%% nodeInterval = bounds . nodes
%% \end{code}
%
A \emph{language} for interaction nets
consists of a type of node labels
together with arity and polarity information defining \emph{all} ports
for each node label, and for any ``function'' node label |f|
and any ``constructor'' node label |c| that can occur as ``argument'' to |f|
a rule, specified by a right-hand side |ruleRHS f c|,
which needs to be a net description having a source compatible with the
\emph{auxiliary} ports of |f|, and a target compatible with the auxiliary ports of |c|.

\begin{code}
data INetLang nLab = INetLang  { polarity  :: ! (nLab -> Vector Polarity)
                               , ruleRHS   :: ! (nLab -> nLab -> NetDescription nLab)
                               }
\end{code}


% Local Variables:
% folded-file: t
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% fold-internal-margins: 0
% end:
