%%%  src/INet/Polar/INet.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{Mutable Net Representation}\sectlabel{INet.Polar.INet}

\begin{ModuleHead}
\begin{code}
module INet.Polar.INet where

import Control.Concurrent.MVar
import INet.Utils.Vector (Vector)
import INet.Polarity (Polarity, opposite)
\end{code}
\end{ModuleHead}

A connection between two ports is implemented as a single |MVar|
that is either empty, or
contains the constructor node for which the connection is at the principal port.
(To allow different node label types to be used, we use the type variable |nLab| throughout.)
%
\begin{code}
type Conn nLab = MVar (Node nLab)
\end{code}
%
For an auxiliary port of a node,
besides its connection
we also record the port's polarity to make it available efficiently at run-time.
(In Haskell, data constructors for simple record types
 habitually are given the same name as the type constructor;
 the fields |pol| and |conn| here are declared strict using ``|!|'', and
 the ``|UNPACK|'' pragma declares an ``unpacking'' optimisation
 as desired to the compiler.)
%
% data Port nLab = Port
%   { pol   :: ! Polarity
%   , conn  :: {-# UNPACK #-} ! (Conn nLab)
%   }
% data Port   nLab = Port  { pol   ::                 ! Polarity
%                          , conn  :: {-# UNPACK #-}  ! (Conn nLab)    }
\begin{code}
data Port nLab = Port
  { pol   ::                 ! Polarity
  , conn  :: {-# UNPACK #-}  ! (Conn nLab)
  }
\end{code}
We introduce the type synonym |Ports| to abbreviate the type of port arrays.
\begin{code}
type Ports  nLab = Vector (Port nLab)
\end{code}
%
Given a port |p|, the port at the other end of its connection
is obtained as |opPort p| by flipping the polarity:
%
% opPort :: Port nLab -> Port nLab
\begin{code}
opPort :: Port nLab -> Port nLab
opPort p = p { pol = opposite $ pol p}
\end{code}
%
A node contains a label,
and the array of its \emph{non-principal} ports.
We do not include the principal port in |ports| since
\begin{itemize}
\item
the prinicipal port of a constructor is connected to the |MVar|
  pointing back to the constructor,
  and
\item
 the prinicipal port of a function is connected to the |MVar|
  the function's thread is waiting on.
\end{itemize}
%
\begin{code}
data Node nLab = Node
  { label :: nLab
  , ports :: Ports nLab
  }
\end{code}

%\edcomm{WK}{Check what happens when a connection consisting of two non-active ports
%has both ports as parts of a redex!}

