%%%  src/INet/Polar/Reduce.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{Interaction Net Reduction}\sectlabel{INet.Polar.Reduce}

\begin{ModuleHead}
\begin{code}
{-# LANGUAGE ScopedTypeVariables, RecursiveDo #-}
module INet.Polar.Reduce where

import INet.Polarity (Polarity(..))
import INet.Polar.INet
import INet.Description
import INet.Utils.MVar

import INet.Utils.Vector  (atErr)
import qualified INet.Utils.Vector as V

import Control.Concurrent
\end{code}
% import Control.Monad (when)
\end{ModuleHead}

The main purpose of the function |replaceNet| is to implement
the instantiation part of the rule application step.
It is a separate function because it also serves the
secondary purpose of constructing the start net.

The function |replaceNet| takes as arguments a |NetDescription|
(defined in \sectref{INet.Description})
for the rule's RHS,
and arrays |src| and |trg| containing the non-principal connections
of the two nodes of the image of rule's LHS in the mutable net representation
(\sectref{INet.Polar.INet}) of the run-time state.

The |mdo| is a ``recursive do'' as introduced by
\citet{Erkoek-Launchbury-2002},
and the use here essentially corresponds to the imperative
programming pattern of allocating an array of uninitialised cells,
and creating references to the array cells
possibly before initialising them.
(Functions prefix with ``\textsf{V.}'' operate on |Vector|s.)

% (Show nLab, Eq nLab} => -- for |debugging|
%
%%     -- PRELIMINARY: check array bounds consistency
%%     let check sx x sy y = when (x /= y) $
%%           fail $ unwords ["replaceNet:", sx, "=", show x, "---", sy, "=", show y]
%%       in do
%%         check "sourceInterval" (sourceInterval descr) "bounds" (bounds src)
%%         check "targetInterval" (targetInterval descr) "bounds" (bounds trg)
\savecolumns
\begin{code}
replaceNet  :: forall nLab . INetLang nLab -> NetDescription nLab
            -> Ports nLab -> Ports nLab -> IO ()
replaceNet lang descr src trg = mdo
    nps <- let  mkNode  (NodeDescription lab pds) = do
                        ps <- V.zipWithM mkPort (polarity lang lab) pds
                        return  ( Node {label = lab , ports = V.tail ps }
                                , V.head ps
                                )
                   where  mkPort Pos  (InternalPort _ _)  = fmap (Port Pos) newEmptyMVar
                          mkPort _    ptd                 = return (portTarget ptd)
                in V.mapM mkNode (nodes descr)
\end{code}
The first step above creates |descr| image nodes,
taking over interface ports from |src| and |trg|,
creating new internal connections at positive ports,
and lazily connecting negative ports with internal connections
located via the function |portTarget| defined below.

Note that the prose explanations here
% before the definition of |reduce| below
are interspersed within the scope of the |mdo| above,
since all code before the definition of |reduce| below remains indented below the |mdo|.
\restorecolumns
\begin{code}
    let  portTarget :: PortTargetDescription -> Port nLab
         portTarget (SourcePort  i) = atErr "portTarget: SourcePort S" src (pred i)
         portTarget (TargetPort  i) = atErr "portTarget: TargetPort S" trg (pred i)
         portTarget (InternalPort n i) = let  e = "portTarget: InternalPort "
                                              (n', pp) = atErr e nps n
           in opPort (if i == 0 then pp else atErr (e ++ shows n " S") (ports n') (pred i))
\end{code}
We traverse the newly created nodes and ``connect'' their principal ports.
%(For internal connections, we could do this already at allocation time in |mkPort|
% above, but then would still need to do it separately for interface connections.)
\restorecolumns
\begin{code}
    let  doNode (n@(Node lab prts), Port pl c) = case pl of
             Neg -> forkIO (reduce lang (ruleRHS lang lab) c prts) >> return ()
             Pos -> putMVar c n
       in V.mapM_ doNode nps
\end{code}
For source and target ports, we only need to take care of short-circuits:
\restorecolumns
\begin{code}
    let  doIfacePort (Port Pos c) ptd = return ()  -- will be done from the other side if necessary
         doIfacePort (Port Neg c) ptd = let        -- original port of the LHS node
             Port _pl' c' = portTarget ptd         -- connecting port in image of RHS
           in if c == c'  then return ()           -- empty cycle
                          else case ptd of
                            InternalPort n i'  -> return ()  -- already dealt with
                            _                  -> do  forkIO (moveMVar c c')
                                                      return ()
       in do  V.zipWithM_ doIfacePort src $ source descr
              V.zipWithM_ doIfacePort trg $ target descr
\end{code}
%  -- putStrLn $ unwords $ map (show . label .fst) $ V.toList nps -- TRACING
%
Whenever a function node is created,
i.e., a node with positive principal port,
a |reduce| thread is started (via |forkIO|).
This thread waits on the connection (|pconn|) between the principal ports of the rule
until this contains the constructor node
(the principal port of which has positive polarity).
The array |src| contains the auxiliary ports of the function node
(the principal port of which has negative polarity).
\begin{code}
reduce :: INetLang nLab -> (nLab -> NetDescription nLab) -> Conn nLab -> Ports nLab -> IO ()
reduce lang rules pconn src = do
   Node clab trg <- takeMVar pconn
   replaceNet lang (rules clab) src trg
\end{code}

%% \begin{code}
%% reduce :: (-- Show nLab, Eq nLab -- for |debugging|
%%   ) => INetLang nLab
%%     -> nLab  -- \edcomm{WK}{for |debugging|}
%%     -> (nLab -> NetDescription nLab)
%%     -> Conn nLab -> Ports nLab -> IO ()
%% reduce lang flab rules pconn src = do
%%   -- putStrLn $ "Starting " ++ show flab
%%   Node clab trg <- takeMVar pconn
%%   -- putStrLn $ "Triggering " ++ show (flab, clab)
%%   replaceNet lang (rules clab) src trg
%%   -- putStrLn $ "Finished   " ++ show (flab, clab)
%% \end{code}

% Local Variables:
% folded-file: t
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% fold-internal-margins: 0
% end:
