%%%  src/INet/Utils/Vector.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{Vector Utilities}

This module is the import interface for |Data.Vector|,
so we have a single place for switching between safe and interface functions.

\begin{code}
module INet.Utils.Vector
  ( module INet.Utils.Vector
  , V.length
  , V.head
  , V.tail
  , V.cons
  , (V.++)
  , V.imap
  , V.foldr
  , V.ifoldr
  , V.zip
  , V.empty
  , V.singleton
  , V.generate
  , V.replicateM
  , V.mapM
  , V.mapM_
  , V.fromList
  , V.toList
  , V.unzip
  , V.zipWithM
  , V.zipWithM_
  , Vector()
  ) where

import Data.Vector (Vector)
import qualified Data.Vector as V
import Control.Monad (liftM2)
\end{code}

\begin{code}
(!) :: Vector a -> Int -> a
(!) = (V.!)
-- (!) = V.unsafeIndex

(!?) :: Vector a -> Int -> Maybe a
(!?) = (V.!?)
-- v !? i = Just (v ! i)
\end{code}

\begin{code}
bounds :: Vector a -> (Int, Int)
bounds v = (0, pred (V.length v))

atErr :: String -> Vector a -> Int -> a
-- atErr s = (V.!)
atErr s v i = case v V.!? i of
  Nothing -> error $ s ++ show i
  Just x -> x
\end{code}

\begin{code}
assocs :: Vector a -> [(Int, a)]
assocs = V.toList . V.indexed
\end{code}

% ifoldr :: (Int -> a -> b -> b) -> b -> Vector a -> b
\begin{code}
imapM' :: Monad m => (Int -> a -> m b) -> Vector a -> m [b]
imapM' f v = V.ifoldr' (\ i a m -> liftM2 (:) (f i a) m) (return []) v
\end{code}
(The $\eta$-expansion via |v| reduces allocation and time quite drastically.
However, $\eta$-expanding the argument function to |V.ifoldr'|
does not seem to make a significant difference.)

|imapM| is more expensive, due to the intermediate vector.
\begin{code}
imapM :: Monad m => (Int -> a -> m b) -> Vector a -> m (Vector b)
imapM f v = V.sequence (V.imap f v)
\end{code}

\begin{code}
pick :: Int -> Vector a -> (a , Vector a)
pick i v = let
    (pre, post) = V.splitAt i v
  in (V.head post, pre V.++ V.tail post)
\end{code}
