-- Copyright 2006 by Wolfram Kahl, all rights reserved

module SImPLEval where

import SImPL

import qualified Data.Map as Map
import Data.Map (Map) -- only the type constructor is imported unqualified



data Value1 = ValInt Integer | ValBool Bool
type State1 = Map Variable Value1

evalExpr   :: Expression -> State1 -> Maybe Value1
evalExpr (Var v) s = Map.lookup v s
evalExpr (Value (LitInt  i)) s = Just (ValInt  i) -- better: function litToVal
evalExpr (Value (LitBool b)) s = Just (ValBool b)
evalExpr (Binary (MkArithOp op) e1 e2) s =
  case (evalExpr e1 s, evalExpr e2 s) of
    (Just (ValInt v1), Just (ValInt v2)) -> evalArithOp op v1 v2
    _                                    -> Nothing
evalExpr (Binary (MkRelOp op) e1 e2) s =
  case (evalExpr e1 s, evalExpr e2 s) of
    (Just (ValInt v1), Just (ValInt v2)) -> evalRelOp op v1 v2
    _                                    -> Nothing
evalExpr (Binary (MkBoolOp op) e1 e2) s =
  case (evalExpr e1 s, evalExpr e2 s) of
    (Just (ValBool b1), Just (ValBool b2)) -> evalBoolOp op b1 b2
    _                                    -> Nothing
evalExpr (Unary Not e) s = case evalExpr e s of
  Just (ValBool b) -> Just (ValBool (not b))
  _                -> Nothing

totalOp mkVal op x y = Just (mkVal (op x y))
totalIntOp = totalOp ValInt
totalBoolOp = totalOp ValBool

evalArithOp Plus  = totalIntOp (+)
evalArithOp Minus = totalIntOp (-)
evalArithOp Times = totalIntOp (*)
evalArithOp Div   = \ x y -> if y == 0 then Nothing
                                       else Just (ValInt (x `div` y))

evalRelOp Less = totalBoolOp (<)
evalRelOp LessOrEqual = totalBoolOp (<=)
evalRelOp Equal = totalBoolOp (==)
evalRelOp NotEqual = totalBoolOp (/=)
evalRelOp Greater = totalBoolOp (>)
evalRelOp GreaterOrEqual = totalBoolOp (>=)

evalBoolOp And = totalBoolOp (&&)
evalBoolOp Or = totalBoolOp (||)

interpStmt :: Statement  -> (State1 -> Maybe State1)

interpStmt (Assignment var e) = \ s -> case evalExpr e s of
  Just val -> Just (Map.insert var val s)
  Nothing  -> Nothing
interpStmt (Conditional cond sThen sElse) = \ s -> case evalExpr cond s of
    Just (ValBool True ) -> (interpStmt sThen) s
    Just (ValBool False) -> (interpStmt sElse) s
    _                    -> Nothing
interpStmt (Loop cond body) = \ s -> case evalExpr cond s of
    Just (ValBool False) -> Just s
    Just (ValBool True ) -> case interpStmt body s of
        Just s1 -> interpStmt (Loop cond body) s1
        _ -> Nothing
    Just (ValInt i) -> Nothing
    Nothing -> Nothing

interpStmt (MkBlock stmts) = \ s -> interpBlock stmts s

interpBlock :: [Statement] -> (State1 -> Maybe State1)
interpBlock [] = Just
interpBlock (stmt : stmts) = \ s -> case interpStmt stmt s of
  Just s1 -> interpBlock stmts s1
  Nothing -> Nothing

interpProg :: Program -> Maybe State1
interpProg (MkProgram decls stmts) = interpBlock stmts Map.empty



instance Show Value1 where
  show (ValInt k) = show k
  show (ValBool b) = show b

showVal :: (Variable, Value1) -> String
showVal (var,val) = var ++ " = " ++ show val

showState1 = unlines . map showVal . Map.toList


runProg :: Program -> String
runProg prog = case interpProg prog of
  Just s -> showState1 s
  Nothing -> "<<< execution failed >>>"
  
