 
// FILE odesolv.h 

#ifndef  INCLUDED_ODESOLV_H
#define  INCLUDED_ODESOLV_H


#include "vndflags.h"
#include "vndstat.h"
#include "odenum.h"
#include "stepctrl.h"
#include "datarepr.h"


class ODE_SOLVER;

/// Pointer to an ODE_SOLVER object.
typedef ODE_SOLVER*   PtrODESolver;


/** Abstract solver class.
  
  @author Ned Nedialkov
  @date 1 May 2001
  */
class ODE_SOLVER 
{
  
public:
  
  /** Integrates an IVP problem.

    This function performs the integration of the problem that is set
    in the constructor.
    
    @param tEnd endpoint where an enclosure of the
    solution is desired
    
    @param Continue if \c true, the integration is from the current
    integration point; otherwise, it is from the initial point. By
    default, \c Continue is set to \c false.
    
    @pre tEnd != ODE->GetT0()
    */
  virtual void IntegrateTo ( double tEnd, bool Continue = false ) = 0;
  
  /** Integrates an ODE_NUMERIC problem.
    
    This function executes \verbatim IntegrateTo( ODE->GetTEnd(),
    Continue ) \endverbatim
    
    @param Continue if \c true, the integration is from the current
    point; otherwise, it is from the initial point.  By default,
    this parameter is set to \c false.  */
  void Integrate ( bool Continue = false );
  
  /** Return a pointer to a DATA_REPR object. 

    Given a pointer to an ODE_SOLVER object, this function casts the
    pointer to its data representation object to a pointer of type
    \c T.
    
    @param Solver a pointer to an ODE_SOLVER object 
    @pre NotNull(Solver)
    */
  template<class T> friend T* GetDataRepr( const PtrODESolver Solver );
  
  /** Sets a data representation object.
    
    @param Data pointer to a data representation object
    @pre NotNull(Data);
    */
  void SetDataRepr ( const PtrDataRepr Data );  

  /** Sets a stepsize control object.
      
      @param Step a pointer to an object that implements a stepsize control.
      @pre NotNull(Step)
  */
  void SetStepControl ( const PtrStepCtrl  Step  );    
  
  /// Returns a pointer to the stepsize control object.
  const PtrStepCtrl GetStepCtrl() const;
  
  
  /** Indicates if verbose output is desired.  
      
      @param b \c true if verbose output is desired and \c false
      otherwise. 
      @see PrintData
  */
  void VerboseOutput ( bool b );
  
  /** Is verbose output desired. 

    This function returns the value that is set by the most recent
    call to \c VerboseOutput, or \c true, if \c VerboseOutput has not
    been called.  */
  bool VerboseOutput () const;   
  
  /**  Prints data. 

    This function outputs data in the following format (the numbers
    are from integrating a test problem)
       
    \verbatim
    *** Integrated from [0,0] to [10,10]
    *** Global Error    1.20e-07
    *** User Time       2.20e-01(sec) 
    *** Accepted Steps  54
    *** Rejected Steps  0
    \endverbatim   
 
    If \c VerboseOutput(false) is called before an integration, such
    an output is not produced.
    
    @param os output stream */
  void PrintData( ostream & os  = cout );
  
  /** Indicates if plots will be generated. 
      
      @param b \c true if files with data are to be generated and
      \c false otherwise.  
      */
  void GraphOutput ( bool b ); 
  
  /** Is graph output desired.
    
    This function returns the value that is set by the most recent
    call to \c GraphOutput, or \c false, if \c GraphOutput has not been called.
    */
  bool GraphOutput () const; 
    
  /** Indicates if the plots are to be extended. 

    @param b \c true if the plots are to be extended and \c false
    otherwise.
    */
  void ContinueOutput( bool b ); 
  
  /** Is graph output continued. 

    This function returns the value that is set by the most recent
    call to \c ContinueOutput, or \c false, if \c ContinueOutput has
    not been called.  */
  bool ContinueOutput() const; 
    
  /** Plots the stepsize versus time. 
      
    This function should be called after an integration has been
    completed, and \c GraphOutput(true) had been called before the
    integration starts.
      
    If a \c file_name is specified, the plot will be stored into an
    encapsulated postscript file under the file name that is given as
    a parameter. Otherwise, the plot will be displayed on the screen.
      
    @param file_name pointer to a string */
  void DisplayStepSize ( const char * file_name = 0 ) const ;
  
  /** Plots the order versus time. 
      
    This function should be called after an integration has been
    completed, and \c GraphOutput(true) had been called before the
    integration starts.
      
    If a \c file_name is specified, the plot will be stored into an
    encapsulated postscript file under the file name that is given
    as a parameter. Otherwise, the plot will be displayed on the
    screen.
    
    @param file_name pointer to a string */
  void DisplayOrder ( const char *file_name = 0 ) const ;
  
  /** Is the first step. 
      @return \c true if a solver is in its first integration step and
      @return \c false otherwise.
  */
  bool FirstStep() const; 
  
  /** Is a step accepted. 
      @return \c true if the solver has accepted the most recent step and
      @return \c false otherwise
  */
  bool AcceptedStep() const; 
  
  
  /// Destructor
  virtual ~ODE_SOLVER();  

  
protected:
  
  /** Constructor. 
    
    @param ODE   pointer to an object that represents the numerical problem
    @param Data  pointer to a data representation object
    @param Step  pointer to an object that implements stepsize control
    @param Flags pointer to an object that updates control flags
    @param Stats pointer to an object that updates statistics
    
    @pre NotNull(ODE) && NotNull(Data) && NotNull(Step) && NotNull(Flags) && NotNull(Stats) 
    */
  ODE_SOLVER( const PtrODENumeric ODE, 
	      const PtrDataRepr   Data, 
	      const PtrStepCtrl   Step,
	      const PtrSolverFlags Flags = new SOLVER_FLAGS,
	      const PtrSolverStats Stats = new SOLVER_STATS );
  
  
  /** Initializes a solver.  */
  virtual void InitSolver() = 0;
  
  
  /** Sets errors. 
    
    This function sets relative and absolute error during an
    integration.
    
    @param t integration point
    @param Y tight enclosure of the solution at \c t.  */
  void SetErrors( double t, const INTERVAL_VECTOR & Y );
  
protected:
  
  /// Pointer to an object representing the numerical problem
  PtrODENumeric  ODE;
  
  /** Pointer to an object that contains a data representation object */
  PtrDataRepr    DataRepr;
  
  /// Pointer to an object that implements stepsize control
  PtrStepCtrl    StepControl;  
  
  /// Pointer to an object that updates control flags
  PtrSolverFlags   SolverFlags;
  
  /// Pointer to an object that updates statistics 
  PtrSolverStats   SolverStats;
  
};



template<class T> T* GetDataRepr( const PtrODESolver Solver )
{
  assert(Solver);
  return static_cast<T*>( Solver->DataRepr );
}


inline const PtrStepCtrl ODE_SOLVER :: GetStepCtrl() const
{
  return StepControl;
}


inline ODE_SOLVER :: ODE_SOLVER( const PtrODENumeric  ODE,
 				 const PtrDataRepr  Data, 
				 const PtrStepCtrl  Step,
				 const PtrSolverFlags Flags,
				 const PtrSolverStats Stats ) 
  : ODE(ODE), 
    DataRepr(Data), 
    StepControl(Step),
    SolverFlags( Flags ),
    SolverStats( Stats )
{ 
  assert( NotNull(ODE) );
  assert( NotNull(Data) );
  assert( NotNull(Step) );
  assert( NotNull(Flags) );
  assert( NotNull(Stats) );
}


inline void ODE_SOLVER :: SetDataRepr ( const PtrDataRepr Data ) 
{ 
  assert(Data);
  DELETE(DataRepr);
  DataRepr = Data; 
}


inline void ODE_SOLVER :: Integrate( bool Continue )
{
  IntegrateTo( ODE->GetTend(), Continue );
}
  

inline ODE_SOLVER :: ~ODE_SOLVER() 
{ 
  DELETE(ODE); 
  DELETE(DataRepr); 
  DELETE(StepControl); 
  DELETE(SolverFlags);
  DELETE(SolverStats);
}


inline void ODE_SOLVER :: SetStepControl  ( const PtrStepCtrl Step ) 
{
  assert( NotNull(Step) );
  DELETE(StepControl); 
  StepControl = Step; 
}


inline void ODE_SOLVER :: GraphOutput( bool b )
{
  SolverFlags->GraphOutput(b);
}


inline bool ODE_SOLVER :: GraphOutput() const 
{
  return  SolverFlags->GraphOutput();
}  


inline void ODE_SOLVER :: VerboseOutput( bool b )
{
  SolverFlags->VerboseOutput( b );
}


inline bool ODE_SOLVER :: VerboseOutput() const 
{ 
  return SolverFlags->VerboseOutput();
}


inline void ODE_SOLVER :: ContinueOutput( bool b )
{
  assert( GraphOutput() && b );
  SolverFlags->ContinueOutput( b );
}


inline bool ODE_SOLVER :: ContinueOutput() const 
{ 
  return SolverFlags->ContinueOutput();
}


inline bool ODE_SOLVER :: FirstStep() const 
{ 
  return SolverFlags->FirstStep();
}


inline bool ODE_SOLVER :: AcceptedStep() const 
{ 
  return SolverFlags->AcceptedStep();
}


#endif
  




