
// FILE vndflags.h 


#ifndef INCLUDED_VNDFLAGS_H
#define INCLUDED_VNDFLAGS_H


#include "vndobj.h"
#include <stdio.h>


/** Flags class. 

  This class provides several flags that are needed by a
  solver.
  
  @author Ned Nedialkov 
  @date   8 June 2001 */
class SOLVER_FLAGS 
{
public: 
  
  /// Constructor 
  SOLVER_FLAGS(); 
  
  /// Reset flags 
  void Reset();
  
  /** Indicates if the step is the first one. 
    
    @param b \c true if the integrator is performing the first
    step and \c false otherwise.
    */
  void FirstStep( bool b ); 
  
  /** Is the first step. 

    This function returns the value that is set by the most recent
    call to \c FirstStep, or \c true, if \c FirstStep has not been
    called.  */
  bool FirstStep ()  const; 
  
  /** Indicates if a step is accepted.  
    
    @param b \c true if a step is accepted and \c false otherwise. 
    */
  void AcceptedStep( bool b );
  
  /**Is the last step accepted. 

    This function returns the value that is set by the most recent
    call to \c AcceptedStep, or \c false, if \c AcceptedStep has not
    been called.  */
  bool AcceptedStep () const; 
  
  /** Indicates if plots will be generated. If the parameter is \c true,
      then files with data will be generated.
      
      @param b \c true if files with data are to be generated and
      \c false otherwise */
  void GraphOutput( bool b ); 
  
  /** Is graph output desired. 

    This function returns the value that is set by the most recent
    call to \c GraphOutput, or \c false, if \c GraphOutput has not
    been called.  */
  bool GraphOutput() const; 
  
  /** Indicates if verbose output is desired.
    
    @param b \c  true if verbose output is desired and \c false
    otherwise. 
    */
  void VerboseOutput( bool b );
  
  /** Is verbose output desired. 
    
    This function returns the value that is set by the most recent
    call to \c VerboseOutput, or \true, if \c VerboseOutput has not
    been called.  */
  bool VerboseOutput() const; 
  
  /** 
    Indicates if we are continuing the graph output. 
    
    @param b \c true if the graph output is to be continued, and
    \c false otherwise.  
    */
  void ContinueOutput( bool b ); 
  
  /** Is graph output continued. This function returns the value that
    is set by the most recent call to \c ContinueOutput, or \c false,
    if \c ContinueOutput has not been called. 
    */
  bool ContinueOutput() const; 


  /** 
    Indicates if we are continuing the integration.
    
    @param b \c true if the integration is continued, and \c false otherwise.  
    */
  void ContinueIntegration( bool b ); 
  
  /** Is the integration continued. 

    This function returns the value that is set by the most recent
    call to \c ContinueIntegration, or \c false, if \c
    ContinueIntegration has not been called.  */
  bool ContinueIntegration() const; 
  
private: 
  
  bool vo, co, go, acc, fs, ci;
  
};


/// Pointer to a SOLVER_FLAGS object
typedef SOLVER_FLAGS * PtrSolverFlags;


inline SOLVER_FLAGS :: SOLVER_FLAGS() 
{
  Reset();
}

inline void SOLVER_FLAGS :: Reset()
{
  vo = true;
  co = false;
  go = false;
  acc = false;
  fs = true;
  ci = false;
}


inline void SOLVER_FLAGS :: FirstStep( bool b ) 
{ 
  fs = b;
}

inline bool SOLVER_FLAGS :: FirstStep() const 
{ 
  return fs;
} 


inline void SOLVER_FLAGS :: AcceptedStep( bool b )  
{ 
  acc = b;
}


inline bool SOLVER_FLAGS :: AcceptedStep() const 
{ 
  return acc;
} 


inline void SOLVER_FLAGS :: GraphOutput( bool b )
{
  go = b;
}


inline bool SOLVER_FLAGS :: GraphOutput() const 
{
  return go;
}  


inline void SOLVER_FLAGS :: VerboseOutput( bool b )
{
  vo = b;
}


inline bool SOLVER_FLAGS :: VerboseOutput() const 
{ 
  return vo;
}


inline void SOLVER_FLAGS :: ContinueOutput( bool b )
{ 
  co = b;
}


inline bool SOLVER_FLAGS :: ContinueOutput() const
{
  return co;
}


inline void SOLVER_FLAGS :: ContinueIntegration( bool b )
{ 
  ci = b;
}


inline bool SOLVER_FLAGS :: ContinueIntegration() const
{
  return ci;
}


#endif

