
// FILE hoe.cc 

#include "hoinit.h"
#include "vnderror.h"
#include "vndutils.h"
#include "taylexp.h"

double CompStepH( const INTERVAL_VECTOR & A, 
		  const INTERVAL_VECTOR & B, int n );


bool HOE :: 
Validate( INTERVAL_VECTOR & Yinit, 
	  INTERVAL_VECTOR & RemTerm,
	  double & h , 
	  int Order, 
	  PtrODENumeric ODE, 
	  PtrODESolver Solver )
{
  assert(Order>0 && Order<=GetMaxOrder());
  assert( NotNull(ODE) );
  assert( NotNull(Solver) );

  // Get the previous enclosure.
  const INTERVAL_VECTOR & Y0 = ODE->GetTightEncl();
  
  // Compute the tolerance.
  double Eps = MaxNorm(Y0)*ODE->GetRtol() + ODE->GetAtol();
  
  // Get the Taylor series generator.
  PtrTaylExp TaylExp = GetDataRepr<TAYL_EXP>(Solver);
  
  // Generate the Taylor terms with Y0 and h.
  if ( Solver->FirstStep() )
    TaylExp->IntSol->GenerateTerms( Y0, h, Order ); 
  else 
    TaylExp->IntSol->GenerateTerms( Y0, h, Order -1 ); 
  
  // Evaluate the Taylor series.
  INTERVAL One( 0.0,1.0 );
  TaylExp->IntSol->GetOneTerm( P, Order - 1 );
  for ( int i = Order - 2; i >= 0; i-- )
    {
      TaylExp->IntSol->GetOneTerm(Y, i);
      // P = Y + One*P;
      P *= One;
      P += Y;
    }
  
  // Guess an enclosure for the remainder term over [t_j,t_j+h].
  if ( Solver->FirstStep() )
    {
      TaylExp->IntSol->GetOneTerm( RemTerm, Order );
      U = INTERVAL(-2.0,2.0)*RemTerm;
      Yinit = P+U;
      
      TaylExp->RemTerm->GenerateTerms( Yinit, h, Order );	  
      TaylExp->RemTerm->GetOneTerm( RemTerm, Order );	  
      U = INTERVAL(-2.0,2.0)*RemTerm;
    }
  else
    {
      assert(hPrev!=0);
      double Ph = 2.0*Power(h/hPrev, Order);
      U = INTERVAL(-Ph,Ph)*PrevLocErr;
    }
  
  // If some of the components of U is the zero interval, then set
  // this interval to [-Eps,Eps].
  for ( int i = 1; i <= ODE->GetSize(); i++ )
    if ( Inf(U(i)) == 0 && Sup(U(i)) == 0 )
      U(i) = INTERVAL(-Eps,Eps);
  
  // Form the initial enclosure.
  Yinit = P + U;
  
  // Generate the remainder term.
  TaylExp->RemTerm->GenerateTerms( Yinit, INTERVAL(h,h), Order );	  
  TaylExp->RemTerm->GetOneTerm( RemTerm, Order );	  
  
  V = One*RemTerm;
  
  if ( !(V <= U) )
    { 
      double alpha = CompStepH( V, U, ODE->GetSize() );
      alpha = Inf( Power( INTERVAL(alpha), DivBounds(1.0, Order) ) );
      assert(alpha<=1);
      
      if ( h > 0 )
	BiasRoundDown();
      else
	BiasRoundUp();
      
      h = alpha*h;
      assert( h != 0 );
      
      if ( fabs(h) < Solver->GetStepCtrl()->CompMinStepSize( ODE->GetTcur(), ODE->GetTend() ) )
	{
	  VnodeMessage("File %s, Line %d : %s\n", __FILE__,__LINE__,
		       "\n The stepsize is too small."
		       "\n Cannot Validate Existence and Uniqueness");
	  return false;
	}
      
      RemTerm *= Power( INTERVAL(alpha,alpha), Order );
      
    }
  
  hPrev = h;
  PrevLocErr = Abs(RemTerm);
  
  return true;
}


void HOE :: ResizeSpace( int N )
{
  Resize(PrevLocErr,N);
  Resize(U,N);
  Resize(P,N);
  Resize(Y,N);
  Resize(V,N);
}


double CompStepH ( const INTERVAL & A, const INTERVAL & B )
{
  // Compute h such that [0,h]*A <= B 
  // B is symmetric.
  assert(INTERVAL(0.0)<=B);  
  
  if ( (Inf(A) == Sup(A)) && (Inf(A) == 0.0))
    return 2.0;
  
  BiasRoundDown();
  
  double h;
  if ( Inf(A) >= 0 )
    {
      assert(Sup(A)!=0);
      // [0,h*Sup(A)] <= [Inf(B), Sup(B)]
      h = Sup(B)/Sup(A);
    }
  else
    if ( Sup(A) <= 0 )
      // [h*Inf(A), 0] <= [Inf(B), Sup(B)]
      h =  Inf(B)/Inf(A);
    else
      //  [h*Inf(A), h*Sup(A)] <= [Inf(B), Sup(B)]
      h = Min( Sup(B)/Sup(A), Inf(B)/Inf(A) ); 
  
  if ( h == 0.0 ) 
    VnodeError( "%s:%d : The stepsize is too small. Cannot compute an enclosure of the solution.",
		__FILE__, __LINE__ );
  assert(h>0);
  return h;
}


double CompStepH( const INTERVAL_VECTOR & A, 
		  const INTERVAL_VECTOR & B, int n )
{
  // Compute h such that [0,h]*A <= B.
  double hmin = CompStepH(A(1),B(1));
  double h;
  for ( int i = 2; i <= n; i++ )
    {
      h = CompStepH( A(i), B(i) );
      if ( h < hmin )
	hmin = h;
    }
  assert(hmin>0);
  return hmin;
}


void HOE :: Init( PtrODENumeric  ODE )
{ 
  ResizeSpace(ODE->GetSize()); 
  hPrev = 0; 
}  






