
// FILE odeprobl.h


#ifndef  INCLUDED_ODEPROBL_H
#define  INCLUDED_ODEPROBL_H


#include "vndinl.h"
#include "vndobj.h"
#include "odesol.h"


/** Pointer to a function for computing the right side of
  \f${y^\prime = f(y)}\f$.
  
  @param Y  pointer to an array of intervals for storing \f${y}\f$.
  @param YP pointer to an array of intervals for storing \f${y^\prime = f(y)}\f$.  */
typedef void (*FCN)( INTERVAL *yp, const INTERVAL *y );

/** Pointer to a function for computing a closed-form solution. 

  If the problem that is integrated has a closed-form solution, then
  the user can set a function for computing it. Such a function is
  used mainly for debugging, to compare the enclosure on the true
  solution computed by the numerical method with the enclosure
  computed by this function.
  
  @param t0 initial time
  @param Y0 initial condition
  @param t  interval where an enclosure is desired, normally a point interval
  
  @param Yexact enclosure of the true solution at \c t 
  */
typedef void  (*EXACT_SOL)( INTERVAL_VECTOR & Yexact, 
			    const INTERVAL & t,
			    const INTERVAL_VECTOR & Y0, 
			    double t0 );


/** Specification of the IVP.
  
  This class specifies the initial condition and the function for
  computing the right side.
    
  @author Ned Nedialkov 
  @date 27 April 2001 
  */
class ODE_PROBLEM : private NAME
{
  
public:
  
  NAME::GetName;
  
  /** Constructor.
      
      @param n    size of the problem
      @param fcn    pointer to a function for computing the right side
      @param name name for the ODE problem

      @pre   n > 0 && NotNull(fcn) && !name.empty() 
  */
  ODE_PROBLEM( int n, FCN fcn, const string & name );

  /// Returns the problem size.
  int GetSize() const;  
  
  /// Sets the initial time, \f${t_0}\f$.
  void SetT0( double t0 );
    
  /** Sets an initial condition given by an interval vector.  

    This function sets \f${[}y_0{]}\f$ at \f${t_0}\f$. If the initial
    condition is a point vector, it has to be stored in an interval
    vector with each component being a point interval.  
    
    @pre GetSize() == Dimension(Y) */
  void SetInitCond( const INTERVAL_VECTOR & Y ); 

  /** Sets a pointer to an initial condition object. 
    
    This object can be of type SOLUTION, or of type derived from
    solution, e.g.  PARALL.  

    @param IC pointer to an object for representing the initial
    condition.
    @pre NotNull(IC) && GetSize() == IC->GetDim() */
  void SetInitCond( const PtrSolution IC ); 
  
  /** Returns the initial time.
    
    This function returns the value for \f${t_0}\f$ that is set in the
    most recent call to \ref SetT0, or 0, if \ref SetT0 has not been
    called.  */
  double  GetT0() const;
  
  /** Returns the interval vector enclosing the initial condition.  */
  const INTERVAL_VECTOR & GetInitCond() const;
  
  /// Returns a pointer to the initial condition object. 
  const PtrSolution GetPtrInitCond() const;
    
  /** Indicates if the problem is constant coefficient.  
    
    @param cc \c true for a constant coefficient problem and \c false
    otherwise.  
    */
  void  ConstCoeff( bool cc );
    
  /** Does the problem have an exact solution.
    
    @return \c true if a function for computing an exact solution is set and
    @return \c false otherwise
    */
  bool  ExactSol()      const;
  
  /** Is the initial condition a point vector.
    
      @return \c true if a point initial condition is specified and 
      @return \c false otherwise
  */
  bool  PointInitCond() const;
  
  /** Is the problem scalar.
    
    @return \c true if the problem is scalar and 
    @return \c false otherwise
  */   
  bool  ScalarProblem() const;
    
  /** Is the problem constant coefficient.
    @return \c true if the problem is constant coefficient and 
    @return \c false otherwise
    */
  bool  ConstCoeff()    const;
				   
  /** Function for computing the right side.  This function computes
    the right side of \f${y^\prime = f(y)}\f$ by calling the function
    specified in the constructor.
    
    @param Y the argument in \f${f(y)}\f$.
    @param YP the result \f${y^\prime = f(y)}\f$
    */
  void CompRightSide( INTERVAL_VECTOR & YP, const INTERVAL_VECTOR & Y ) const;
  
  /** Sets a function for computing an exact solution.

    @param f pointer to a function for computing an exact solution
    @pre NotNull(f)
    */
  void SetCompExactSol( EXACT_SOL f );
  
  /** A function for computing an exact solution.  

    This function computes an enclosure on the true solution by
    calling the function that is set in \ref SetCompExactSol.
    
    @param t interval (normally a point interval) where an exact solution is desired
    @param Y enclosure of the exact solution. 
    
    */
  void CompExactSol( INTERVAL_VECTOR & Y, const INTERVAL & t );
  
  /// Destructor.
  virtual ~ODE_PROBLEM();
  
private:
  
  /// size of the problem
  int   Size;             
  
  /// pointer to a function for computing the right side
  FCN   Fcn;             
  
  /// pointer to the initial condition
  PtrSolution  InitCond;     
  
  /// pointer to a function for computing a closed form solution
  EXACT_SOL CompSol;         
  
  /// information about the ODE problem
  class PROBLEM_INFO
  {
  public:
    /// is it a constant coefficient problem
    bool ConstCoeff;   
    
    /// it it a scalar problem 
    bool Scalar;       
    
    /// does it have an exact solution   
    bool ExactSol;     
    
    /// is a point initial condition specified
    bool PointInitCond;
    
    /// Constructor
    PROBLEM_INFO() : ConstCoeff(false), Scalar(false), 
      ExactSol(false), PointInitCond(true) {;}
  } 
  ProblemInfo; 
  
};



inline void ODE_PROBLEM :: SetT0( double t0 ) 
{ 
  assert( NotNull(InitCond) );
  InitCond->SetTcur(t0); 
}


inline void ODE_PROBLEM :: SetInitCond( const PtrSolution IC ) 
{ 
  assert( NotNull(IC) );
  assert( GetSize() == IC->GetDim() );
  DELETE(InitCond); 
  InitCond = IC; 
}


inline void  ODE_PROBLEM :: SetCompExactSol( EXACT_SOL f ) 
{ 
  assert(NotNull(f)); 
  CompSol = f; 
}


inline int ODE_PROBLEM :: GetSize() const 
{ 
  return Size;
}


inline double ODE_PROBLEM :: GetT0() const    
{ 
  assert( NotNull(InitCond) ); 
  return InitCond->GetTcur(); 
}


inline const INTERVAL_VECTOR & ODE_PROBLEM :: GetInitCond() const 
{ 
  assert( NotNull(InitCond) );
  return InitCond->GetTightEncl(); 
}


inline const PtrSolution ODE_PROBLEM :: GetPtrInitCond() const 
{ 
  assert( NotNull(InitCond) );
  return InitCond;
}


inline void ODE_PROBLEM :: ConstCoeff( bool cc ) 
{ 
  ProblemInfo.ConstCoeff = cc; 
}


inline bool ODE_PROBLEM :: ExactSol() const       
{ 
  return CompSol == NULL? false:true; 
}


inline bool ODE_PROBLEM :: PointInitCond() const  
{ 
  return  ProblemInfo.PointInitCond; 
}


inline bool ODE_PROBLEM :: ScalarProblem() const  
{ 
  return ProblemInfo.Scalar; 
}  


inline bool ODE_PROBLEM :: ConstCoeff() const  
{ 
  return ProblemInfo.ConstCoeff; 
}


inline void  ODE_PROBLEM :: CompExactSol( INTERVAL_VECTOR & Y, 
					  const INTERVAL & t ) 
{ 
  assert(CompSol); 
  CompSol(Y,t, GetInitCond(), GetT0() ); 
}


inline void ODE_PROBLEM::
CompRightSide( INTERVAL_VECTOR & YP, const INTERVAL_VECTOR & Y ) const
{
  assert(Fcn);
  Fcn( THE_ELEM(YP), THE_ELEM(Y) );  
}


inline ODE_PROBLEM :: ~ODE_PROBLEM()  
{ 
  DELETE(InitCond); 
}


#endif 
