
// FILE odesol.h 


#ifndef INCLUDED_SOLUTION
#define INCLUDED_SOLUTION


#include <assert.h>
#include "vnodecnf.h"


/** Base solution class.

  The \c SOLUTION class specifies the current and previous integration
  points, \f$t_{j}\f$ and \f${t_{j-1}}\f$, respectively, an enclosure
  \f$[y_{j}]\f$ of the solution at \f$t_{j}\f$ and an a priori
  enclosure \f$[\tilde{y}_{j-1}]\f$ of the solution on
  \f$[t_{j-1},t_{j}]\f$.


  @author Ned Nedialkov 
  @date   7 June 2001 
  */
class SOLUTION 
{

public:
  
  /** Constructor. 
      
      @param n size of the problem 
      @pre n > 0
  */
  SOLUTION( int n );

  /// Returns the value of \c n set in the constructor.
  int GetDim() const;
  
  /** Sets a current point. 

      This function sets the current integration point. Its previous
      value is saved and can be obtained by calling \ref GetTprev.
    
      @param t current point.  */
  void SetTcur( double t );
  
  /** Sets a tight enclosure. 
      
      This function sets an interval vector that contains a tight
      enclosure of the solution.
    
      @param Y enclosure of the solution.
      @pre GetDim() == Dimension(Y) */
  void SetTightEncl( const INTERVAL_VECTOR & Y );
  
  /** Sets an priori enclosure. 

      This functions sets an a priori enclosure of the solution over
      the interval formed from the previous and current integration
      points.

      @pre GetDim()==Dimension(Y) 
  */
  void SetInitEncl( const INTERVAL_VECTOR & Y );
  
  /** Returns the current integration point. 
      
      This function returns the value that is set by the most recent
      call to \ref SetTcur, or \c 0, if \ref SetTcur has not been
      called.  */
  double GetTcur() const;
  
  /** Returns the previous integration point. 

      This function returns the value that is set by the second most
      recent call to \ref SetTcur, or \c 0, if \ref SetTcur has been
      called at most once.  
  */
  double GetTprev() const; 
  
  /** Returns the tight enclosure. 

      This function returns the interval vector that is set by the
      most recent call to \ref SetTightEncl.  */
  const INTERVAL_VECTOR & GetTightEncl() const; 
  
  /** Returns the a priori enclosure. 

      This function returns the interval vector that is set by the
      most recent call to \ref SetInitEncl.  */
  const INTERVAL_VECTOR & GetInitEncl() const; 

  /** Assignment operator. 
    @param S reference to a solution object
    @pre  GetDim()==S.GetDim();
    */
  SOLUTION & operator = ( const SOLUTION & S);
  
  /// Destructor.  
  virtual ~SOLUTION(); 

private:

  /// current and previous integration points 
  double tCur, tPrev;
  
  ////enclosure of the solution at the current integration
  //point, tCur
  INTERVAL_VECTOR Ytight; 
  
  /// a priori enclosure on [tPrev,tCur]
  INTERVAL_VECTOR Yinit; 
  
};


/// Pointer to a SOLUTION object.
typedef SOLUTION*  PtrSolution;  


inline SOLUTION :: SOLUTION( int n ) : tCur(0), tPrev(0)
{ 
  assert( n>0 ); 
  
  Resize( Ytight, n ); 
  Resize( Yinit, n ); 
}


inline void SOLUTION :: SetTcur( double t )  
{ 
  tPrev = tCur;
  tCur = t;
}

inline void SOLUTION :: SetTightEncl( const INTERVAL_VECTOR & Y ) 
{ 
  assert( GetDim()==Dimension(Y) );
  Ytight = Y; 
}


inline void SOLUTION :: SetInitEncl( const INTERVAL_VECTOR & Y ) 
{ 
  assert( GetDim()==Dimension(Y) );
  Yinit = Y; 
}


inline double SOLUTION :: GetTcur() const    
{ 
  return tCur; 
}


inline double SOLUTION :: GetTprev() const    
{ 
  return tPrev; 
}


inline const INTERVAL_VECTOR & SOLUTION :: GetTightEncl() const 
{ 
  assert(Dimension(Ytight)!=0);
  return Ytight; 
}


inline const INTERVAL_VECTOR & SOLUTION :: GetInitEncl() const 
{ 
  return Yinit; 
}


inline SOLUTION :: ~SOLUTION()
{
  ;
}


inline int SOLUTION :: GetDim() const
{
  return Dimension(Ytight);
}


#endif
