
// FILE orderctrl.h 


#ifndef INCLUDED_ORDCTRL_H
#define INCLUDED_ORDCTRL_H


#include "odesolv.h"


/** Abstract order control class.
  
  @author Ned Nedialkov
  @date 23 April 2001
  */ 
class ORDER_CTRL : private NAME
{
  
public:
  
  NAME::GetName;
  
  /** Constructor. It sets the maximum order allowed, \c kmax, the
    value for the current order to \c kmax, and a name for the order
    control method.
    
    @param kmax maximum order allowed
    @param name name for the order control method
    
    @pre kmax > 0 && !name.empty() */
  ORDER_CTRL( int kmax, const string & name );

  /** Computes order. 

     @param Solver pointer to a solver 

     @return a value for the order of a method
     @pre NotNull(Solver)
     */
  virtual int CompOrder( PtrODESolver Solver ) = 0;

  /** Gets the maximum order. 

      This function returns the maximum value for the order that is
      set in the constructor. */
  int GetMaxOrder() const; 

  /** Sets the current order. 

      This function stores a value for the current order.

      @param k order 
      @pre k > 0 && k <= GetMaxOrder() 
  */
  void SetCurOrder( int k );

  /** Gets the current order. 

      This function returns the order that is set by the most recent
      call to \ref SetCurOrder or the order that is set in the
      constructor, if \ref SetCurOrder has not been called.  
  */
  int GetCurOrder() const;
  
  /// Destructor
  virtual ~ORDER_CTRL() {;}
  
  
private:

  /// current order
  int CurOrder;
  
  /// maximum order
  int MaxOrder;

};


/// Pointer to an order control object
typedef ORDER_CTRL* PtrOrderCtrl;


inline ORDER_CTRL :: ORDER_CTRL( int kmax, const string & name ) : 
  NAME(name), 
  CurOrder(kmax), MaxOrder(kmax) 
{
  assert( kmax>0 );
}


inline int ORDER_CTRL :: GetMaxOrder() const 
{ 
  assert( MaxOrder>0 );  
  return MaxOrder;  
}


inline void ORDER_CTRL :: SetCurOrder( int k ) 
{ 
  assert( k>0 && k <= GetMaxOrder() );
  CurOrder = k;  
}


inline int ORDER_CTRL :: GetCurOrder() const
{ 
  assert( CurOrder > 0 );
  return CurOrder;
}

#endif 
