
// FILE paral.h


#ifndef INCLUDED_PARALL_H
#define INCLUDED_PARALL_H


#include "odesol.h"
#include "vndinl.h"

/** Parallelepiped representation.

  The PARALL class specifies a parallelepiped in the form \f${\{
  \,y_j+A_jr_j \mid r_j \in [r_j] \, \}}\f$, where \f${y_j}\f$ is an
  \f${n}\f$ dimensional point vector, \f${A_j}\f$ is an \f${n\times
  n}\f$ point matrix, and \f${[r_j]}\f$ is an \f${n}\f$ dimensional
  interval vector.
  
  @author Ned Nedialkov 
  @date 7 June 2001 
*/
class PARALL : public SOLUTION
{

public:
  
  /** Constructor. 

      @param n size of the parallelepiped
      @pre n>0 
  */
  PARALL( int n );
  
  /** Constructor.  
    
    It creates a parallelepiped with the specified parameters.  This
    constructor also sets a tight enclosure by calling 
    \c SOLUTION::SetTightEncl( \c Y+A*R \c ); 

    \see \ref SOLUTION::SetTightEncl.
    
    @param Y point vector 
    @param A point (transformation) matrix
    @param R interval vector
    
    @pre Square(A,Dimension(R)) && Dimension(Y)== Dimension(R)
    */
  PARALL( const VECTOR & Y, const MATRIX & A, const INTERVAL_VECTOR & R );
  
  /** Sets a point vector. 
    
    This function sets the point vector \f${y_j}\f$ in the parallelepiped
    \f${\{ \,y_j+A_jr_j \mid r_j \in [r_j] \, \}}\f$.
    
    @param Y  point vector
    @pre GetDim()==Dimension(Y) 
    */
  void SetPointVec ( const VECTOR & Y );
    
  /** Sets a transformation matrix.
    
    This function sets the point matrix \f${A_j}\f$ in the
    parallelepiped \f${\{ \,y_j+A_jr_j \mid r_j \in [r_j] \, \}}\f$.
    
    @param A point matrix
    
    @pre Square(A, GetDim() )
    */
  void SetMatrix ( const MATRIX & A );
  
  /** Sets an interval vector. This function sets the interval vector
      in the parallelepiped \f${\{ \,y_j+A_jr_j \mid r_j \in [r_j] \,
      \}}\f$.
      
      @param R  interval vector
      @pre GetDim() == Dimension(R) */
  void SetIntervalVec ( const INTERVAL_VECTOR & R );
  
  /** Sets a parallelepiped. 

    This function sets a parallelepiped with the specified parameters and also 
    sets a tight enclosure by calling \c
    SOLUTION::SetTightEncl( \c Y+A*R \c ); 
    
    \see \ref SOLUTION::SetTightEncl.


    @param Y point vector 
    @param A point (transformation) matrix
    @param R interval vector
    
    @pre Square(A,Dimension(R)) && Dimension(R)== Dimension(Y) */
  void SetParal( const VECTOR & Y, const MATRIX & A, 
		 const INTERVAL_VECTOR & R );
  
  /** Sets a parallelepiped. 

    This function sets a parallelepiped in the form of a box with
    edges parallel to the axes of the coordinate system.

    @param Y interval vector
    @pre GetDim() == Dimension(Y) */
  void SetParal( const INTERVAL_VECTOR & Y );
  
  /** Returns the point vector. 

      This function returns the most recently set point vector in the
      parallelepiped.  */
  const VECTOR & GetPointVec()  const;

  /** Gets a matrix. 

      This function returns the most recently set point matrix in the
      parallelepiped.  */
  const MATRIX & GetMatrix()    const;
    
  /** Gets an interval vector. 

      This function returns the most recently set interval vector in
      the parallelepiped.  */
  const INTERVAL_VECTOR & GetIntervalVec() const;
  
  /** Assignment operator. 

      It converts the tight enclosure from the solution object to a
      parallelepiped representation.
            
      @param S a reference to a solution object.  
      @pre GetDim() == S.GetDim() */
  PARALL & operator=( const SOLUTION & S );
  
  /// Destructor.
  virtual ~PARALL();
  

private:
  
  /// point vector 
  VECTOR Yj;      
  
  /// transformation matrix
  MATRIX Aj;           
  
  /// error vector
  INTERVAL_VECTOR Rj;  
  
  /// resize space
  void ResizeSpace( int n );
  
};


/// Pointer to a parallelepiped object
typedef PARALL*  PtrParall;


inline PARALL :: PARALL( int n ) : SOLUTION( n ) 
{ 
  assert( n>0 );
  ResizeSpace( n );
}


inline void PARALL :: SetPointVec( const VECTOR & Y )
{ 
  assert( GetDim() == Dimension(Y) );
  Yj = Y; 
}


inline void PARALL :: SetMatrix( const MATRIX & A ) 
{ 
  assert( Square(A,GetDim()) );  
  Aj = A; 
}


inline void PARALL :: SetIntervalVec( const INTERVAL_VECTOR & R ) 
{ 
  assert( GetDim() == Dimension(R));
  Rj = R; 
}


inline const VECTOR & PARALL :: GetPointVec() const 
{ 
  return Yj; 
}


inline const MATRIX & PARALL :: GetMatrix() const 
{ 
  return Aj; 
}


inline const INTERVAL_VECTOR & PARALL :: GetIntervalVec() const 
{ 
  return Rj;
}

  
inline PARALL :: ~PARALL() 
{
  ;
}

#endif
