
// FILE stepctrl.h 

#ifndef  INCLUDED_STEPCTRL_H
#define  INCLUDED_STEPCTRL_H


#include "odenum.h"
#include "datarepr.h"


/** Abstract stepsize control class. 
  
  @author Ned Nedialkov 
  @date   8 June 2001
  */ 
class STEP_CTRL : private NAME
{
  
public:
  
  NAME::GetName;
  
  /** Constructor. It sets a name for the stepsize control method.
    
    @param name name for the stepsize control method
    
    @pre !name.empty() */
  STEP_CTRL( const string & name );
  
  /** Sets a stepsize. This function stores the absolute value of a stepsize.
    
    @param h stepsize. 
    @pre h != 0.0
    */
  void SetStepSize( double h );
  
  /** Returns the current stepsize.  This function returns the
    stepsize that is set by the most recent call to \ref SetStepSize,
    or \c 0, if \ref SetStepSize has not been called.  */
  double GetStepSize () const;

  /** Computes the first step.
    
    This function computes a stepsize on the first integration step.
    
    @param ODE pointer to problem
    @param Data pointer to a data representation object
    @param order order of the method
    
    @return the predicted initial stepsize 

    @pre NotNull(ODE) && NotNull(Data) && order > 0 */
  virtual double PredictFirstStep( const PtrODENumeric ODE, 
				 PtrDataRepr Data, 
				 int order ) = 0;
  
  /** Computes a new stepsize. 

    This function computes a stepsize after the first integration step.

    @param hPred  (output) predicted stepsize 
    @param h      current stepsize
    @param ODE    pointer to a problem
    @param LocErr enclosure of the local error on the current step
    @param Ytight tight enclosure of the solution 
    @param order  order of the method
    
    @return \c true if the stepsize \c h is accepted and 
    @return \c false if the stepsize \c h is rejected
 
    @pre h != 0 && NotNull(ODE) && Dimension(LocErr) == ODE->GetSize() &&
    @pre  Dimension(Ytight) == ODE->GetSize() && order > 0
    */
  virtual bool PredictStep( double & hPred, double h, 
			    const PtrODENumeric    ODE, 
			    const INTERVAL_VECTOR  & LocErr, 
			    const INTERVAL_VECTOR  & Ytight, 
			    int order ) = 0;
  
    /** Computes the minimum stepsize allowed. 
    
    @param t current integration point
    @param T end of the integration interval 

    @pre t != T
    @return the minimum stepsize allowed
    */
  double CompMinStepSize( double t, double T );  
    
  /** Computes the next point. 
    
    @param h  current stepsize
    @param t  current integration point
    @param T  end of the integration interval 
    
    @return the next integration point
    
    @pre h != 0 && t != T
    */
  double CompNextPoint( double h, double t, double T );
  
  /** Compute the last stepsize. 

    If \c t is "too close" to \c T, this function reduces \c h and returns
    the reduced stepsize.  */
  double IfLastStep( double t, double h, double T );
  
  /// Initializes a stepsize control object
  virtual void Init( const PtrODENumeric ODE ) = 0;  

  /// Destructor  
  virtual ~STEP_CTRL() {};
  
protected:

  STEP_CTRL() { hCur = 0; }

private:

  /// stepsize 
  double hCur; 
  
};


/// Pointer to a STEP_CTRL object.
typedef STEP_CTRL* PtrStepCtrl;


inline STEP_CTRL :: STEP_CTRL( const string & name )
  : NAME( name )
{ 
  ;
}


inline void STEP_CTRL :: SetStepSize( double h ) 
{ 
  assert( h != 0 );  
  hCur = fabs(h); 
}


inline double STEP_CTRL :: GetStepSize() const 
{ 
  return hCur; 
}


#endif 
