
// FILE taylctrl.h 


#ifndef INCLUDED_TAYLCTRL_H
#define INCLUDED_TAYLCTRL_H 


#include "vnodecnf.h"
#include "vndinl.h"


/** Control class for Taylor coefficient generators. 

    This class specifies the order of a Taylor series expansion, the
    maximum order allowed, and the point of expansion.
    
    @author Ned Nedialkov 
    @date   8 June 2001 */
class TAYLOR_CTRL 
{
  
public:
  
  /** Constructor. 

    @param n size of the problem
    @pre  n > 0
    */
  TAYLOR_CTRL( int n );
  
  /** Sets a point of expansion. It is specified by an interval
      containing this point.
      
      @param h an interval containing the point of expansion
      @pre NonZero(h)
  */
  void SetH    ( const INTERVAL & h );
  
  /** Returns the interval enclosing the point of expansion. This
      function returns the interval that is set by the most recent
      call to \ref SetH, or \c 0, if \ref SetH has not been called. 
  */
  INTERVAL  GetH ()  const;
  
  /** Sets the order of the Taylor series.
      @param order order of the Taylor series
      @pre   order>0 && order<= GetMaxOrder()
  */
  void SetOrder( int order );
    
  /** Returns the order of the Taylor series. 

      This function returns the order that is set by the most recent
      call to \ref SetOrder, or \c 0, if \ref SetOrder has not been
      called. 
  */
  int  GetOrder ()  const;
  
  /// Returns the maximum order allowed.
  static int  GetMaxOrder ();
  
  /// Returns the size of the problem.
  int  GetDim   ()  const;
  
  /// Destructor
  virtual ~TAYLOR_CTRL();

  
public:
  
  /** Maximum order of the Taylor series. A default value of 40 is set
      in the \ref VnodeInit function. This value can be changed by editing 
      the file \c vndinit.cc.
  */
  static int MaxOrder;
  

private:
  
  /// point of expansion
  INTERVAL  H;    
  
  /// dimension of the problem
  int  Dim;  
  
  /// order of the expansion
  int  Order;    
}; 


inline TAYLOR_CTRL :: TAYLOR_CTRL( int n )
  : H(0.0), Dim(n), Order(0) 
{
  assert( n>0 );
}


inline void TAYLOR_CTRL :: SetH  ( const INTERVAL & h ) 
{ 
  assert( NonZero(h) );
  H = h; 
}


inline INTERVAL TAYLOR_CTRL :: GetH ()  const 
{ 
  return H; 
}


inline void TAYLOR_CTRL :: SetOrder( int order )     
{ 
  assert( order>0 && order<=GetMaxOrder() );
  Order = order; 
}


inline int  TAYLOR_CTRL :: GetMaxOrder ()   
{ 
  return MaxOrder; 
}


inline int  TAYLOR_CTRL :: GetOrder ()  const
{ 
  return Order; 
}


inline int  TAYLOR_CTRL :: GetDim   ()  const
{ 
  return Dim; 
}


inline TAYLOR_CTRL :: ~TAYLOR_CTRL() 
{
  ;
}


#endif
