
// FILE tgtencl.h 


#ifndef INCLUDED_TGTENCL_H
#define INCLUDED_TGTENCL_H


#include "odesolv.h"


/**  Abstract class for computing tight enclosures on the solution.

  @author Ned Nedialkov
  @date   8 June 2001
*/
class TIGHT_ENCL : protected NAME
{
public: 
  
  using NAME::GetName;
  
  /** Computes a tight enclosure. 
      
      This function computes a tight enclosure of the solution at
      \verbatim ODE->GetTcur() + h \endverbatim
    
      @param Ytight  (output)  enclosure of the solution
      @param LocErr  (output) enclosure of the local error
      @param h       enclosure of the stepsize
      @param order   order of the enclosure method
      @param ODE     pointer to a problem
      @param Solver  pointer to a solver

      @pre NonZero(h) && order>0 && order<=GetMaxOrder() && NotNull(ODE) && NotNull(Solver) &&
      ODE->GetSize() == Dimension(Ytight) && 
      ODE->GetSize() == Dimension(LocErr) */
  virtual void CompEncl( INTERVAL_VECTOR & Ytight, 
			 INTERVAL_VECTOR & LocErr,
			 const INTERVAL & h, 
			 int order,
			 PtrODENumeric ODE, 
			 PtrODESolver Solver ) = 0;

  /** Initializes the enclosure method.
      
      @param ODE pointer to a problem
      @pre NotNull(ODE)  */

  virtual void Init( const PtrODENumeric ODE ) = 0;  

  /** Returns the maximum order allowed. 
      
      This function returns the value that is set in the constructor of
      \c TIGHT_ENCL. */
  int GetMaxOrder()  const;

  /// Destructor
  virtual ~TIGHT_ENCL() {;}
  

protected:
  
  /** Constructor. 
    
    @param max_order maximum order allowed
    @param name name of the method
    
    @pre max_order>0 && !name.empty() 
  */
  TIGHT_ENCL( int max_order, const string & name );
  
  /// Rearranges the columns of A.
  void RearrangeColumns( MATRIX & R, 
			 const MATRIX & A, 
			 const INTERVAL_VECTOR & R );

private:
  
  /// Maximum order allowed. 
  const int MaxK;
  
};


/// Pointer to a TIGHT_ENCL object
typedef TIGHT_ENCL* PtrTightEncl;


inline TIGHT_ENCL :: TIGHT_ENCL( int max_order, const string & name ) 
  : NAME(name), MaxK(max_order)
{
  assert( max_order > 0 );
}


inline int TIGHT_ENCL :: GetMaxOrder() const
{
  return MaxK;
}


#endif
