
// FILE varstep.cc 

#include "vnderror.h"
#include "vndutils.h"
#include "varstep.h"
#include "taylexp.h"


bool VAR_STEP_CTRL :: 
PredictStep( double & h_new, 
	     double h_old, 
	     const PtrODENumeric  ODE, 
	     const INTERVAL_VECTOR & LocErr, 
	     const INTERVAL_VECTOR & Ytight, 
	     int Order )
{
  assert(h_old!=0);
  assert(Order>1);
  
  // We use a mixture of absolute and relative error control.  
  BiasRoundNear();
  
  double Tol = ODE->GetRtol()*MaxNorm(Ytight) + ODE->GetAtol();   
  double Error = MaxNorm( Diam(LocErr) );
  
  bool accepted;
  
  if ( Error == 0 )
    {
      h_new = 2*h_old;
      return true;
    }
  
  double h = fabs(h_old);
  Tol *= h;   // local error control per unit step.
  
  if ( Error <= Tol )
    {
      accepted = true;
      h_new = 0.9*h*Power(0.5* Tol/Error, 1.0/(Order-1) );
      if (h_new > 2*h)
	h_new = 2*h;
    }
  else 
    {
      accepted = false;
      h_new = 0.9*h*Power( Tol/Error, 1.0/(Order-1) );
      assert(h_new<h);
    }

  if ( h_new < CompMinStepSize( ODE->GetTcur(), ODE->GetTend() ) )
    {
      VnodeMessage("File %s, Line %d : %s\n", 
		   __FILE__, __LINE__, 
		   "\n The stepsize is too small."
		   "\n Possibly the enclosures are too wide");
    }
  return accepted;
}


double VAR_STEP_CTRL :: 
PredictFirstStep( const PtrODENumeric  ODE, 
		  PtrDataRepr  DataRepr, int Order ) 
{
  // Select an initial stepsize. The idea is borrowed from
  // Eijgenraam's thesis.
  
  const INTERVAL_VECTOR & Ytight = ODE->GetInitCond();
  double Tol = ODE->GetAtol()+ODE->GetRtol()*MaxNorm(Ytight);
  
  PtrTaylExp p = static_cast<TAYL_EXP*>(DataRepr);
  
  p -> IntSol->GenerateTerms( Ytight, INTERVAL(1.0), Order+1 );
  p->IntSol->GetOneTerm( Tmp, Order+1 );
  
  const double TmpNorm = MaxNorm(Tmp);  
  double h;  
  if (TmpNorm == 0)
    h = 2.0;
  else
    {
      BiasRoundNear();
      h = Tol/((Order+1)*TmpNorm );
      h = Power(h, (1.0/Order));
    }
  
  return h/2;
}












