
// FILE vndstat.h


#ifndef INCLUDED_VNDSTATS_H
#define INCLUDED_VNDSTATS_H

#include <assert.h>
#include "vnodecnf.h" 


/** Class for storing the error. 
  
  @author Ned Nedialkov 
  @date   18 June 2001
  */
class ERROR
{

public:
  
  /// Constructor.
  ERROR() : T(0), E(-1) {;}
  
  /** Sets an error.
      @param t integration point
      @param e error at \c t.
      
      @pre e >= 0
      */
  void SetError( double t, double e )  { assert(e>=0); T = t; E = e; }
    
  /** Returns the error.
      @return the error set by the most recent call to \ref SetError or
      \c -1, if \ref SetError has not been called.
  */
  double GetError() const { return E; }
  
  /** Returns the point of the error.  

      @return the point set by the most recent call to \ref SetError,
      or \c 0, if \ref SetError has not been called. */
  double GetT() const { return T; }
  

private:

  /// integration point 
  double T;

  /// error at the integration point
  double E; 
};




/** Statistics class.

  The purpose of this class is to store statistics collected during an
  integration.
  
  @author Ned Nedialkov 
  @date   8 June 2001 */
class SOLVER_STATS
{
  
public:
  
  /// Constructor
  SOLVER_STATS();
  
  /// Resets the statistics.
  void Reset();
  
  /** Sets relative error. This function sets the relative error at a
    given point and also sets the maximum relative error.
      
      @param t integration point
      @param e relative error at \c t
      @pre e >= 0 */
  void SetRelError( double t, double e );
  
  /** Sets absolute error. This function sets the absolute error at a
      given point and also sets the maximum absolute error. 
      
      @param t integration point 
      @param e absolute error at \c t. 
      @pre e >= 0 
  */
  void SetAbsError( double t, const double e );
  
  /** Returns the maximum relative error. 
      
      @return the maximum relative error or \c -1, if \ref SetRelError
      has not been called.  
  */
  double GetMaxRelError() const;
  
  /** Returns the maximum absolute error. 
      
      @return the maximum absolute error or \c -1, if \ref SetAbsError
      has not been called.  
  */
  double GetMaxAbsError() const;
    
  /** Returns the point of the maximum relative error.  

      @return the point where the maximum relative error occurs or \c 0,
      if \ref SetRelError has not been called.  
  */
  double GetWhereMaxRelError() const;
  
  /** Returns the point of the maximum absolute error.  
      
      @return the point where the maximum absolute error occurs or \c 0,
      if \ref SetAbsError has not been called.  
  */
  double GetWhereMaxAbsError() const;
  
  /** Returns the current relative error.
      
      @return the relative error that is set by the most recent call to 
      \ref SetRelError, or \c -1, if \ref SetRelError has not been called.
  */
  double GetRelError() const;
    
  /** Returns the current absolute error.
      
      @return the absolute error that is set by the most recent call to 
      \ref SetAbsError, or \c -1, if \ref SetAbsError has not been called.
  */
  double GetAbsError() const;
  
  /** Returns the point of the current relative error.
      
      @return the point that is set by the most recent call to 
      \ref SetRelError, or \c 0 if \ref SetRelError has not been called.
  */
  double GetWhereRelError() const;
  
  /** Returns the point of the current absolute error.
      
      @return the point that is set by the most recent call to 
      \ref SetAbsError, or \c 0 if \ref SetAbsError has not been called.
  */
  double GetWhereAbsError() const;
  
  /** Adds CPU time. This function adds \c time to the current CPU time.

      @param time CPU time.
      @pre time >= 0
  */
  void AddUserTime( double time );
  
  /// Returns the CPU time
  double GetUserTime()  const;
  
  /// Returns the number of accepted steps
  unsigned int  GetAcceptedSteps() const;
  
  /// Returns the number of rejected steps
  unsigned int  GetRejectedSteps() const;

  /// Returns the number of accepted and rejected steps
  unsigned int  GetTotalSteps() const;
  
  /** Keeps track of the accepted and  rejected steps.
      
      @param b if \c true,  the number of accepted steps is
      incremented by one; otherwise, the number of rejected steps is
      incremented by one.  
  */
  void AccRejStep( bool b );

  
private:
  
  /// number of accepted steps
  unsigned int NofSteps;
  
  /// number of rejected steps
  unsigned int NofRejectedSteps;
  
  /// CPU time
  double   CPUTime;
  
  ///  absolute and relative errors
  ERROR MaxRelError, RelError, MaxAbsError, AbsError;
  
}; 


/// Pointer to a SOLVER_STATS object
typedef SOLVER_STATS * PtrSolverStats;


inline SOLVER_STATS :: SOLVER_STATS()
{
  Reset();
}


inline void SOLVER_STATS :: SetRelError( double t, double e ) 
{ 
  assert( e>=0 );
  
  RelError.SetError( t,e ) ; 
  
  if ( e > MaxRelError.GetError() ) 
    MaxRelError.SetError( t,e ); 
} 


inline double SOLVER_STATS :: GetMaxRelError() const 
{ 
  return MaxRelError.GetError(); 
}


inline double SOLVER_STATS :: GetWhereMaxRelError() const 
{ 
  return MaxRelError.GetT(); 
}


inline double SOLVER_STATS :: GetRelError() const 
{ 
  return RelError.GetError(); 
}


inline double SOLVER_STATS :: GetWhereRelError() const 
{ 
  return RelError.GetT(); 
}


inline void SOLVER_STATS :: SetAbsError( double t, double e ) 
{ 
  assert( e>=0 );
  AbsError.SetError(t,e); 
  
  if ( e > MaxAbsError.GetError() ) 
    MaxAbsError.SetError(t,e); 
} 


inline double SOLVER_STATS :: GetMaxAbsError() const 
{ 
  return MaxAbsError.GetError(); 
}


inline double SOLVER_STATS :: GetWhereMaxAbsError() const 
{ 
  return MaxAbsError.GetT(); 
}


inline double SOLVER_STATS :: GetAbsError() const 
{ 
  return AbsError.GetError(); 
}


inline double SOLVER_STATS :: GetWhereAbsError() const 
{ 
  return AbsError.GetT(); 
}


inline void SOLVER_STATS :: AddUserTime ( double time ) 
{ 
  assert( time>=0 );

  BiasRoundNear();
  CPUTime += time; 
}


inline double  SOLVER_STATS :: GetUserTime () const 
{ 
  return CPUTime; 
}


inline unsigned int SOLVER_STATS ::  GetRejectedSteps() const 
{ 
  return NofRejectedSteps; 
};


inline unsigned int SOLVER_STATS ::  GetAcceptedSteps() const 
{ 
  return NofSteps; 
};


inline unsigned int SOLVER_STATS ::  GetTotalSteps() const 
{ 
  return NofSteps+NofRejectedSteps; 
};


inline void SOLVER_STATS :: AccRejStep( bool b )
{
  if ( b )
    NofSteps++;
  else
    NofRejectedSteps++;
}


#endif
