function [M,B,Q,steps,nVec] = BlkLanCom(A,R,steps)
% [M,B,Q,steps,nVec] = BlkLanCom(A,R,steps)
%
% Lanczos block tridiagonalization of a complex-symmetric matrix.
% Componentwise orthogonalization scheme is applied.
%
% Input
%   A     complex-symmetric matrix.
%   R     starting matrix of orthonormal columns, assuming the number
%         of columns is the block size.
%   steps number of iterations 
% Output
%   M     3-d array, M(:,:,i) are the main diagonal blocks of the
%         resulting  block tridiagonal
%   B     3-d array, B(:,:,i) are the subdiagonal blocks of the
%         resulting block tridiagonal
%   Q     unitary
%   steps number of iterations actually run
%   nVec  number of vectors selected for reorthogonalization 
% so that
%            A = Q*J*Q.'
% where J is block tridiagonal whose main diagonal blocks are M(:,:,i)
% and subdiagonal blocks are B(:,:,i).
%
[n,bs] = size(R);               % get the size of the starting matrix
if steps > n/bs 
    steps = n/bs; 
end
%
% constants
IM = sqrt(-1);
SQRTEPS = sqrt(eps);
MIDEPS = sqrt(sqrt(SQRTEPS))^7;	% between sqrt(eps) and eps
%
Q(:,1:bs) = R;                  % initial block in Q
M = zeros(bs,bs,steps);         % initialize M and B
B = zeros(bs,bs,steps-1);
W = zeros(bs,bs,steps,2);       % orthogonality estimates
%
up = 0;                         % upper bound
doOrtho = 0;			% if do orthogonalization
second = 0;			% if this is the second orthogoalization
nBlk = 0;                       % number of blocks for reorthogonalization
%
% start the first block Lanczos iteration
cur = 1;                        % indices of W for the recursion
old = 2;
qLow = 1;                       % initial low and up indices of Q
qUp = bs;
Tmp = A*conj(Q(:,qLow:qUp));
M(:,:,1) = Q(:,qLow:qUp)'*Tmp;                  % M(1)=Q(1)'*A*conj(Q(1))        
R = Tmp - Q(:,qLow:qUp)*M(:,:,1);               % R(1)=A*conj(Q(1))-Q(1)M(1) 
[Q(:,(qLow+bs):(qUp+bs)), B(:,:,1)] = qr(R, 0); % QR factorization. Q(2)B(1)=R              
W(:,:,1,cur) = (eps*bs*0.6*B(:,:,1) ...         % compute W(:,:,1,2)   
                .*(randn(bs,bs)+IM*randn(bs,bs)))/B(:,:,1);    
%           
% following block Lanczos iterations
cur = 2;
old = 1;
for j = 2:steps-1  
    qLow = qLow + bs;           % update low and up indices of Q            
    qUp = qUp + bs;
    Tmp = A*conj(Q(:,qLow:qUp));
    M(:,:,j) = Q(:,qLow:qUp)'*Tmp;             
    R = Tmp - Q(:,qLow:qUp)*M(:,:,j) - Q(:,(qLow-bs):(qUp-bs))*B(:,:,j-1).';
    [Q(:,(qLow+bs):(qUp+bs)), B(:,:,j)] = qr(R, 0);       
%       
    % block orthogonalization
    if (second == 0)            % not second orthogonalization
        k = 1;
        while ((k <= j) & (doOrtho ~= 1))       % compute W(k,j+1), k = 1,...j
            if (k == j)
                W(:,:,k,old) = (eps*bs*0.6)*(B(:,:,1) ...
                                .*(randn(bs,bs)+IM*randn(bs,bs))); 
            else % k<j
                W(:,:,k,old) = M(:,:,k)*conj(W(:,:,k,cur)) ...
                               - W(:,:,k,cur)*M(:,:,j) ...
                               + (eps*0.3)*((B(:,:,1)+B(:,:,2)) ...
                               .*(randn(bs,bs)+IM*randn(bs,bs)));
                if (j > 2)
                    if (k > 1)
                        W(:,:,k,old) = W(:,:,k,old) ...
                                       + B(:,:,k-1)*conj(W(:,:,k-1,cur));
                    end % if (k>1)
                    if (k < j-1)
                        W(:,:,k,old) = W(:,:,k,old) ...
                                       + ((B(:,:,k).')*conj(W(:,:,k+1,cur)) ...
                                       - W(:,:,k,old)*(B(:,:,j-1).'));
                    end % if (k<j-1)
                end % if (j>2)
            end % if (k=j)
            W(:,:,k,old) = W(:,:,k,old)/B(:,:,j);
%            
            % find the first W which loses orthogonality
            for colW = 1:bs             % for each column of W(:,:,k,j+1)
                if (max(abs(W(:,colW,k,old))) >= SQRTEPS)
                    doOrtho = 1;        % found loss of orthogonalization
                    up = k;
                    break;
                end
            end % for colW
            k = k + 1;
        end % while ((k <= j) & (doOrtho ~= 1)) 
%        
        if ((doOrtho == 1) & (up < j))        
            % if loss of ortho was found and not the last W
            thresh = 0;         % flag if MIDEPS is found           
            k = j;              % search from the last W 
            while ((k >= 2) & (thresh ~= 1))    
                if (k == j)
                    W(:,:,k,old) = (eps*bs*0.6)*(B(:,:,1) ...
                                        .*(randn(bs,bs)+IM*randn(bs,bs)));
                else % k<j
                    W(:,:,k,old) = M(:,:,k)*conj(W(:,:,k,cur)) ...
                                   + B(:,:,k-1)*conj(W(:,:,k-1,cur)) ...
                                   - W(:,:,k,cur)*M(:,:,j) ...
                                   + (eps*0.3)*((B(:,:,k)+B(:,:,j)) ...
                                   .*(randn(bs,bs)+IM*randn(bs,bs)));
                    if (k < j - 1)
                        W(:,:,k,old) = W(:,:,k,old) ...
                                       + (B(:,:,k).')*conj(W(:,:,k+1,cur)) ...
                                       - W(:,:,k,old)*(B(:,:,j-1).');
                    end % if (k<j-1)
                end % if (k=j)
                W(:,:,k,old) = W(:,:,k,old)/B(:,:,j);
%
                % check if W(:,:,k,j+1) exceeds MIDEPS
                for colW = 1:bs         % for each column of W
                    if (max(abs(W(:,colW,k,old))) >= MIDEPS)
                        thresh = 1;     % found a W
                        up = k;
                        break;
                    end 
                end % for colW
                k = k - 1;
            end % while ((k >= 2) & (thresh ~= 1)) 
        end % if ((doOrtho == 1) & (up ~= j))
    else        % second orthogonalization
        if (up < j)             % compute Ws in [up j]
            for k = up:j
                if (k == j)
                    W(:,:,k,old) = (eps*bs*0.6)*(B(:,:,1) ...
                                   .*(randn(bs,bs)+IM*randn(bs,bs)));
                else % k<j
                    W(:,:,k,old) = M(:,:,k)*conj(W(:,:,k,cur)) ...
                                   + B(:,:,k-1)*conj(W(:,:,k-1,cur)) ...
                                   - W(:,:,k,cur)*M(:,:,j) ...
                                   + (eps*0.3)*(B(:,:,k)+B(:,:,j)) ...
                                   .*(randn(bs,bs)+IM*randn(bs,bs));
                    if (k < j-1)
                        W(:,:,k,old) = W(:,:,k,old) ...
                                       +(B(:,:,k).')*conj(W(:,:,k+1,cur)) ...
                                       - W(:,:,k,old)*(B(:,:,j-1).');
                    end % if (k<j-1)
                end % if (k=j)
                W(:,:,k,old) = W(:,:,k,old)/B(:,:,j);
            end % for k = up:j
        end % if (up < j)
    end % if (second == 0)
%
    tmp = old;                  % swap indices old and cur
    old = cur;
    cur = tmp;
%    
    if ((doOrtho == 1) | (second == 1))    
        % orthogonalize Q(j+1) against Q(k), k is inside the interval
        qLow2 = 1 - bs;         % initial low and up indices
        qUp2 = 0;               % for Q which is reorthogonalized
        for (k = 1:up)
            qLow2 = qLow2 + bs;
            qUp2 = qUp2 + bs;
            for (colR = 1:bs)
                for (colQ = qLow2:qUp2)
                    R(:,colR) = R(:,colR) - (Q(:,colQ)'*R(:,colR))*Q(:,colQ);
                end
            end                
            %reset orthogonality estimates
            W(:,:,k,cur) = eps*1.5*(randn(bs,bs) + IM*randn(bs,bs)); 
        end % for (k = 1:up) 
        [Q(:,(qLow+bs):(qUp+bs)), B(:,:,j)] = qr(R, 0);      
        nBlk = nBlk + up;       % update number of blocks selected for 
                                % orthogonalization
%            
        if (second == 1)
            second = 0;
            up = 0;             % clear upper bounds for next iteration
        else
            second = 1;
            doOrtho = 0;
            up = min(j + 1, up + 1);    % adjust for second orthogonalization
        end       
    end % if ((doOrtho == 1) | (second == 1)) 
end % for j = 2:steps-1
%
% the last iteration
qLow = qLow + bs;                   
qUp = qUp + bs;
Tmp = A*conj(Q(:,qLow:qUp));
M(:,:,steps) = Q(:,qLow:qUp)'*Tmp;   
%
nVec = nBlk * bs;               % update the number of vectors
                                % selected for orthogonalization
