function [M,B,Q,steps,nVec] = BlkLanNorm(A,R,steps)
% [M,B,Q,steps,nVec] = BlkLanNorm(A,R,steps)
%
% Lanczos block tridiagonalization of a complex-symmetric matrix.
% Modified partial orthogonalization using 2-norm calculation is 
% applied if required.
%
% Input
%   A     complex-symmetric matrix.
%   R     starting matrix of orthonormal columns, assuming the number
%         if rows is a multiple of the number of columns..
%   steps number of iterations 
% Output
%   M     3-d array, M(:,:,i) are the main diagonal blocks of the
%         resulting  block tridiagonal
%   B     3-d array, B(:,:,i) are the subdiagonal blocks of the
%         resulting block tridiagonal
%   Q     unitary
%   steps number of iterations actually run
%   nVec  number of vectors selected for reorthogonalization 
% so that
%            A = Q*J*conj(Q'),
% where J is block tridiagonal whose main diagonal blocks are M(:,:,i)
% and subdiagonal blocks are B(:,:,i).
%
[n,bs] = size(R);               % get the size of the starting matrix
if steps > n/bs steps = n/bs; end
%
% constants
SQRTEPS = sqrt(eps);
MIDEPS = sqrt(SQRTEPS)^3;       % between sqrt(eps) and eps
%
Q(:,1:bs) = R;                  % initial block in Q
M = zeros(bs,bs,steps);         % initialized M and B; 
B = zeros(bs,bs,steps-1);
a = zeros(steps,1);             % a(j) stores 2-norm of M(:,:,j)
b = zeros(steps,1);             % b(j) stores 2-norm of B(:,:,j)
c = zeros(steps,1);             % c(j) stores 1/min(svd(B(:,:,j))
%
es = eps*bs;                    % error term
wOld = zeros(steps,1);	        % orthogonality estimates
wCur = zeros(steps,1);
wOld(1) = es;
%
up = ones(steps,1);	        % upper and lower bounds for
low = ones(steps,1);            % orthogonalization intervals
interNum = 0;			% orthogonalization interval number
doOrtho = 0;			% if do orthogonalization
second = 0;			% if this is the second partial orthog
nBlk = 0;			% number of blocks selected for orthogonalization
%
qLow = 1 - bs;                  % set initial low and up indices for Q
qUp = 0;
for j = 1:steps  
    qLow = qLow + bs;           % set low, up indices for Q for this step 
    qUp = qUp + bs;
    Tmp = A*conj(Q(:,qLow:qUp));
    M(:,:,j) = Q(:,qLow:qUp)'*Tmp;      % set M(j) = Q(j)'*A*conj(Q(j))   
    a(j) = norm(M(:,:,j));              % set a(j) for error estimate
%    
    % calculate R(j) = A*conj(Q(j)) - Q(j)M(j) - Q(j-1)B(j-1)^T
    if j == 1
        R = Tmp - Q(:,qLow:qUp)*M(:,:,j);
    else    
        R = Tmp - Q(:,qLow:qUp)*M(:,:,j) - Q(:,(qLow-bs):(qUp-bs))*B(:,:,j-1).';
    end
%
    if (j < steps)    
        % QR factorization. Q(j+1)B(j) = R
        [Q(:,(qLow+bs):(qUp+bs)), B(:,:,j)] = qr(R, 0);        
        b(j) = norm(B(:,:,j));          % set b(j), c(j) for error estimate
        c(j) = 1/min(svd(B(:,:,j)));
%        
        if (j > 2) 		        % compute orthogonality estimates
            % compute w(k,j+1) where k = 1,...,j-1
            wOld(1) = (b(1)*wCur(2) + b(j-1)*wOld(1) ...
                       + (a(1) + a(j))*wCur(1))*c(j);
            wOld(2:j-1) = (b(2:j-1).*wCur(3:j) ...
                           + b(1:j-2).*wCur(1:j-2) ...       
                           + b(j-1)*wOld(2:j-1) ...
                           + (a(2:j-1) + a(j)).*wCur(2:j-1))*c(j);
%
            % swap wOld and wCur
            tmp = wOld(1:j-1);
            wOld(1:j-1) = wCur(1:j-1);
            wCur(1:j-1) = tmp;
            wOld(j) = 1.0;
        end % if j>2
        wCur(j) = es;                   % set w(j, j+1)
        wCur(j+1) = 1.0;                % set w(j+1, j+1)
%
        if (second == 0)	        % not the second time, determine intervals
            doOrtho = 0;	        % initialization
            interNum = 0;
            k = 1;
            while k <= j
                if (wCur(k) >= SQRTEPS)	        % lost orthogonality
                    doOrtho = 1;
                    interNum = interNum + 1;
                    % find the upper bound
                    p = k + 1;
                    while ((p < (j + 1)) & (wCur(p) >= MIDEPS))
                        p = p + 1;	        % nearly lost orthogonality
                    end % while
                    up(interNum) = p - 1;
                    % find the lower bound
                    p = k - 1;
                    while ((p > 0) & (wCur(p) >= MIDEPS))
                        p = p - 1;	        % nearly lost orthogonality
                    end % while
                    low(interNum) = p + 1;
                    %
                    k = up(interNum) + 1;       % continue search
                else
                    k = k + 1;
                end % if lost orthogonality
            end % while k <= j
        end % if not second time
%
        if (doOrtho | (second == 1))	        
            % now we have intervals, carry out orthogonalization
            for (k = 1:interNum)		% for each interval
                qLow2 = (low(k) - 2)*bs + 1;    % set initial low and up indices
                qUp2 = qLow2 + bs - 1;          % for Q which is reorthogonalized
                for (i = low(k):up(k))
                    qLow2 = qLow2 + bs;
                    qUp2 = qUp2 + bs;
                    % orthogonalize R against Q(j)
                    for (y = 1:bs)
                        for (x = qLow2:qUp2)
                            R(:,y) = R(:,y) - (Q(:,x)'*R(:,y))*Q(:,x);
                        end
                    end                
                    wCur(i) = es;               % reset ortho estimates
                end % for (i = low(k):up(k))
                %
                nBlk = nBlk + up(k) - low(k) + 1;
                % count the number of vectors selected
                if (second == 1)	        % this is the second time
                    second = 0;		        % reset
                    low(k) = 0; 
                    up(k) = 0; 
                else
                    second = 1;		        % do second time
                    doOrtho = 0;	        % reset
                    % adjust orthogonalization intervals for the second time
                    low(k) = max(1, low(k) - 1);
                    up(k) = min(j + 1, up(k) + 1);
                end % if
            end % for (k = 1:interNum)
%
            % Recalculate QR factorization. Q(j+1)B(j) = R
            [Q(:,(qLow+bs):(qUp+bs)), B(:,:,j)] = qr(R, 0);        
        end % if (doOrtho | (second == 1))
    end % if j < steps 
end % for j = 1:steps
%
nVec = nBlk * bs;
